package com.atlassian.aws.ec2.caches;

import com.amazonaws.services.ec2.AmazonEC2Async;
import com.amazonaws.services.ec2.model.DescribeImagesRequest;
import com.amazonaws.services.ec2.model.Image;
import com.google.common.base.MoreObjects;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;

public class ImageCache
{
    private final AmazonEC2Async ec2Client;
    private final ConcurrentMap<String, Image> cachedImages = Maps.newConcurrentMap();

    public ImageCache(final AmazonEC2Async ec2Client)
    {
        this.ec2Client = ec2Client;
    }

    public List<Image> describeImages(final String... imageIds)
    {
        final List<Image> allImages = new ArrayList<>();
        final Set<String> imagesToDescribe = Sets.newHashSet(imageIds);
        for (final String imageId : imageIds)
        {
            final Image image = cachedImages.get(imageId);
            if (image!=null)
            {
                imagesToDescribe.remove(image.getImageId());
                allImages.add(image);
            }
        }

        if (!imagesToDescribe.isEmpty() || imageIds.length==0)
        {
            final DescribeImagesRequest request = new DescribeImagesRequest().withImageIds(imagesToDescribe);
            final List<Image> describedImages = ec2Client.describeImages(request).getImages();

            allImages.addAll(addToCache(describedImages));
        }

        return allImages;

    }

    private List<Image> addToCache(@NotNull final List<Image> images)
    {
        final List<Image> imagesFromCache = new ArrayList<>();
        for (final Image image : images)
        {
            final Image imageFromCache = cachedImages.putIfAbsent(image.getImageId(), image);
            imagesFromCache.add(MoreObjects.firstNonNull(imageFromCache, image));
        }
        return imagesFromCache;
    }
}
