package com.atlassian.aws.ec2.awssdk.launch;

import com.amazonaws.services.ec2.model.BlockDeviceMapping;
import com.amazonaws.services.ec2.model.EbsBlockDevice;
import com.amazonaws.services.ec2.model.Image;
import com.amazonaws.services.ec2.model.InstanceNetworkInterfaceSpecification;
import com.amazonaws.services.ec2.model.ResourceType;
import com.amazonaws.services.ec2.model.SpotFleetTagSpecification;
import com.amazonaws.services.ec2.model.TagSpecification;
import com.atlassian.aws.AWSAccount;
import com.atlassian.aws.ec2.EC2InstanceType;
import com.atlassian.aws.ec2.EC2Utils;
import com.atlassian.aws.ec2.InstanceLaunchConfiguration;
import com.atlassian.aws.ec2.model.ResourceId;
import com.atlassian.aws.ec2.model.SecurityGroupId;
import com.atlassian.aws.ec2.model.SubnetId;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

public class LauncherUtils {
    static boolean isVpcEnabled(final InstanceLaunchConfiguration instanceConfiguration)
    {
        return !Iterables.isEmpty(instanceConfiguration.getSubnets());
    }

    static Collection<BlockDeviceMapping> getBlockDeviceMappings(@NotNull final AWSAccount awsAccount,
                                                                 @NotNull final String imageId,
                                                                 @NotNull final EC2InstanceType instanceType,
                                                                 @Nullable final String snapshotId,
                                                                 @Nullable final Integer rootFsSize) {
        final Image image = awsAccount.describeImage(imageId);
        Preconditions.checkNotNull(image, "Unable to find " + imageId + ", make sure it exists and your account has permissions to access it");

        final Collection<BlockDeviceMapping> newBdms = new ArrayList<>();
        if (rootFsSize!=null) {
            final Optional<BlockDeviceMapping> presumedRootBdm = image.getBlockDeviceMappings().stream()
                    .filter(bdm -> bdm.getEbs() != null && StringUtils.isNotBlank(bdm.getEbs().getSnapshotId()))
                    .findFirst();
            
            presumedRootBdm.ifPresent(
                    rootBdm -> {
                        final EbsBlockDevice ebs = rootBdm.getEbs()
                                .withVolumeSize(rootFsSize);

                        if (Objects.equals(Boolean.FALSE, ebs.isEncrypted())) {
                            //needs to be explicitely cleared
                            ebs.setEncrypted(null);
                        }
                        newBdms.add(rootBdm);
                    }
            );
        }

        newBdms.addAll(getEphemeralBlockDeviceMappings(image, instanceType));
        if (snapshotId!=null) {
            newBdms.addAll(getEbsBlockDeviceMappings(image, snapshotId));
        }

        return newBdms;
    }

    private static Collection<BlockDeviceMapping> getEphemeralBlockDeviceMappings(@NotNull final Image image, @NotNull final EC2InstanceType instanceType)
    {
        final List<BlockDeviceMapping> bdms = new ArrayList<>();

        //we always set up at least 1 device (may be wrong, backward compatibility)
        //if there are SSDs on an instance, we set up an additional volume
        final int volumesToSetUp = instanceType.getSsdCnt()<2 ? 1 : 2;

        for (int i=0; i<volumesToSetUp ; i++)
        {
            final BlockDeviceMapping ephemeralBlockDeviceMapping =
                    new BlockDeviceMapping()
                            .withDeviceName(getInstanceStoreDeviceName(image, instanceType, i))
                            .withVirtualName("ephemeral" + i);

            bdms.add(ephemeralBlockDeviceMapping);
        }

        return bdms;
    }

    private static String getInstanceStoreDeviceName(final Image image, final EC2InstanceType instanceType, final int index)
    {
        if (!EC2Utils.isWindows(image))
        {
            return "/dev/sd" + getLetterWithIndex(index+1); //b = 0
        }
        if (instanceType != EC2InstanceType.Hs18xlarge)
        {
            //documentation says "xvd[a-e]", but this won't work with some of the instances. When in doubt, ignore docs and use the instance launch wizard.
            return "xvd" + getLetterWithIndex(index+1); //b = 0
        }
        else
        {
            return "xvdc"  + getLetterWithIndex(index); //a = 0
        }
    }

    private static char getLetterWithIndex(final int i)
    {
        return (char)('a' + i);
    }

    private static Collection<BlockDeviceMapping> getEbsBlockDeviceMappings(@NotNull final Image image, @NotNull final String snapshotId)
    {
        final EbsBlockDevice ebs =
                new EbsBlockDevice()
                        .withDeleteOnTermination(true)
                        .withSnapshotId(snapshotId);
        final BlockDeviceMapping blockDeviceMapping =
                new BlockDeviceMapping()
                        .withDeviceName(EC2Utils.getEbsDeviceName(image))
                        .withEbs(ebs);
        return Collections.singletonList(blockDeviceMapping);
    }

    public static InstanceNetworkInterfaceSpecification newPublicIpForVpc(@NotNull final SubnetId subnetId, @NotNull final Iterable<SecurityGroupId> securityGroups)
    {
        return new InstanceNetworkInterfaceSpecification()
                .withAssociatePublicIpAddress(true)
                .withSubnetId(subnetId.getId())
                .withGroups(ResourceId.getIds(securityGroups))
                .withDeleteOnTermination(true)
                .withDeviceIndex(0);
    }

    static TagSpecification newTagSpecification(final InstanceLaunchConfiguration instanceConfiguration) {
        return new TagSpecification().withResourceType(ResourceType.Instance).withTags(instanceConfiguration.getTag());
    }

    static SpotFleetTagSpecification newFleetTagSpecification(final InstanceLaunchConfiguration instanceConfiguration) {
        return new SpotFleetTagSpecification().withResourceType(ResourceType.Instance).withTags(instanceConfiguration.getTag());
    }
}
