package com.atlassian.aws.ec2.awssdk;

import com.amazonaws.regions.Regions;
import com.amazonaws.services.ec2.model.InstanceState;
import com.amazonaws.services.ec2.model.VirtualizationType;
import com.google.common.base.Preconditions;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Map;

/**
 * This class is meant to provide support for AWS SDK constants which are not defined in the SDK itself.
 * These constants should be removed as soon as they are officially introduced in the API.
 * For now, they will reside in this class only (as opposed to separate classes), to have a better overview
 * of stuff we'd rather not have in our code.
 *
 */
public class AwsSupportConstants
{
    public static final String ADDRESS_DOMAIN_VPC = "vpc";

    private AwsSupportConstants()
    {
    }

    public interface NameProvider
    {
        String getName();
    }

    /**
     * NOTE: These enums are persisted to the database. Do not remove or rename them
     */
    public enum Architecture implements NameProvider
    {
        x86_64("x86_64"),
        i386("i386");

        private final String name;

        Architecture(final String awsName)
        {
            this.name = awsName;
        }

        @Override
        @NotNull
        public String getName()
        {
            return name;
        }

        @NotNull
        public static Architecture fromAwsName(@NotNull String awsName)
        {
            return translateToEnum(awsName, values());
        }
    }

    /**
     * NOTE: These enums are persisted to the database. Do not remove or rename them
     */
    public enum Platform implements NameProvider
    {
        linux("linux"),
        windows("windows");

        private final String name;

        Platform(final String awsName)
        {
            this.name = awsName;
        }

        @Override
        @NotNull
        public String getName()
        {
            return name;
        }

        public boolean isWindows()
        {
            return StringUtils.containsIgnoreCase(name, "windows");
        }

        @NotNull
        public static Platform fromAwsName(@NotNull final String awsName)
        {
            return translateToEnum(awsName, values());
        }
    }

    /**
     * NOTE: These enums are persisted to the database. Do not remove or rename them
     */
    public enum Virtualisation implements NameProvider
    {
        UNKNOWN(""),
        PV("pv"),
        HVM("hvm");

        private final String name;

        Virtualisation(final String virtualisationType)
        {
            this.name = virtualisationType;
        }

        @Override
        @NotNull
        public String getName()
        {
            return name;
        }

        @Nullable
        public VirtualizationType toAws()
        {
            switch (this)
            {
                case HVM:
                    return VirtualizationType.Hvm;
                case PV:
                    return VirtualizationType.Paravirtual;
                case UNKNOWN:
                    return null;
                default:
                    throw new IllegalArgumentException("Conversion from " + this + " not supported");
            }
        }

        @NotNull
        public static Virtualisation fromAwsName(@NotNull final String awsName)
        {
            if (awsName.equals("paravirtual"))
            {
                return PV;
            }
            return translateToEnum(awsName, values());
        }
    }

    /**
     * NOTE: These enums are persisted to the database. Do not remove or rename them
     */
    public enum RootDeviceType implements NameProvider
    {
        S3("s3"),
        EBS("ebs");
        private final String name;

        RootDeviceType(final String awsName)
        {
            this.name = awsName;
        }

        @Override
        @NotNull
        public String getName()
        {
            return name;
        }

        @NotNull
        public static RootDeviceType fromAwsName(@NotNull final String awsName)
        {
            if (awsName.equalsIgnoreCase("instance-store"))
            {
                return S3;
            }
            return translateToEnum(awsName, values());
        }
    }

    /**
     * NOTE: These enums are persisted to the database. Do not remove or rename them
     */
    public enum Region implements NameProvider
    {
        CA_CENTRAL_1(Regions.CA_CENTRAL_1, "Canada (Central)"),
        US_EAST_1(Regions.US_EAST_1, "US East (Northern Virginia)"),
        US_EAST_2(Regions.US_EAST_2, "US East (Ohio)"),
        US_WEST_1(Regions.US_WEST_1, "US West (Northern California)"),
        US_WEST_2(Regions.US_WEST_2, "US West (Oregon)"),

        EU_CENTRAL_1(Regions.EU_CENTRAL_1, "EU (Frankfurt)"),
        EU_NORTH_1(Regions.EU_NORTH_1, "EU (Stockholm)"),
        EU_WEST_1(Regions.EU_WEST_1, "EU (Ireland)"),
        EU_WEST_2(Regions.EU_WEST_2, "EU (London)"),
        EU_WEST_3(Regions.EU_WEST_3, "EU (Paris)"),

        ASIA_PACIFIC_E_1(Regions.AP_EAST_1, "Asia Pacific (Hong Kong)"),
        ASIA_PACIFIC_SE_1(Regions.AP_SOUTHEAST_1, "Asia Pacific (Singapore)"),
        ASIA_PACIFIC_SE_2(Regions.AP_SOUTHEAST_2, "Asia Pacific (Sydney)"),
        ASIA_PACIFIC_NE_1(Regions.AP_NORTHEAST_1, "Asia Pacific (Tokyo)"),
        AP_NE_2(Regions.AP_NORTHEAST_2, "Asia Pacific (Seoul)"),
        AP_S_1(Regions.AP_SOUTH_1, "Asia Pacific (Mumbai)"),

        SOUTH_AMERICA_1(Regions.SA_EAST_1, "São Paulo"),
        US_GOV_W1(Regions.GovCloud, "GovCloud (US)"),
        US_GOV_E1(Regions.US_GOV_EAST_1, "GovCloud (US East)"),
        CN_NORTH_1(Regions.CN_NORTH_1, "China (Beijing)"),
        CN_NORTHWEST_1(Regions.CN_NORTHWEST_1, "China (Ningxia)"),
        CUSTOM("custom", "Custom", "custom");

        private final String name;
        private final String displayName;
        private final String endpoint;
        private final com.amazonaws.regions.Region region;

        Region(@NotNull final String name, @NotNull final String displayName, @NotNull final String endpoint)
        {
            this.name = name;
            this.displayName = displayName;
            this.endpoint = endpoint;
            this.region = null;
        }

        Region(@NotNull final Regions region, @NotNull final String displayName)
        {
            this.region = RegionSupport.toRegion(region);
            this.name = RegionSupport.getNormalisedName(region);
            this.displayName = displayName;
            this.endpoint = Ec2Support.getEndpoint(region);
            if (endpoint==null)
            {
                throw new IllegalArgumentException("No endpoint found for region " + region);
            }
        }

        @NotNull
        @Override
        public String getName()
        {
            return name;
        }

        @NotNull
        public String getNormalisedName()
        {
            return name;
        }

        @NotNull
        public String getDisplayName()
        {
            return displayName;
        }

        @NotNull
        public String getEndpoint()
        {
            return endpoint;
        }

        @Nullable
        public com.amazonaws.regions.Region getSdkRegion()
        {
            return region;
        }
    }

    public static final Supplier<Map<String, Region>> LOCATION_CONSTRAINT_TO_REGION = Suppliers.memoize(() ->
    {
        final ImmutableSet<Region> locationConstraintExceptions = ImmutableSet.of(Region.US_EAST_1, Region.EU_WEST_1);

        final ImmutableMap.Builder<String, Region> constraintMap = ImmutableMap.<String, Region>builder()
                .put("US", Region.US_EAST_1)
                .put("EU", Region.EU_WEST_1);

        for (Region region : Region.values())
        {
            if (locationConstraintExceptions.contains(region))
            {
                continue;
            }

            final com.amazonaws.regions.Region sdkRegion = region.getSdkRegion();
            if (sdkRegion==null)
            {
                continue;
            }
            constraintMap.put(sdkRegion.getName(), region);
        }

        return constraintMap.build();
    });

    /**
     * This will translate region code into region enum. Supported region codes: direct AWS S3 and normalised
     * Bamboo codes.
     * @param region
     * @return
     */
    @NotNull
    public static Region translateToRegion(@NotNull final String region)
    {
        final Region regionEnum = LOCATION_CONSTRAINT_TO_REGION.get().get(region);
        if (region.equals("us-east-1"))
        {
            return Region.US_EAST_1;
        }
        return (regionEnum != null)? regionEnum : translateToEnum(region, Region.values());
    }

    @NotNull
    private static <T extends NameProvider> T translateToEnum(@NotNull final String name, T... values)
    {
        for (T someEnum : values)
        {
            if (someEnum.getName().equals(name))
            {
                return someEnum;
            }
        }
        throw new IllegalArgumentException("Unable to translate [" + name + "] to enum " + values.getClass());
    }

    public enum InstanceInitiatedShutdownBehaviour
    {
        STOP("stop"),
        TERMINATE("terminate");
        private final String behaviour;

        InstanceInitiatedShutdownBehaviour(final String behaviour)
        {
            this.behaviour = behaviour;
        }

        @Override
        public String toString()
        {
            return behaviour;
        }
    }

    public enum InstanceAttribute
    {
        INSTANCE_INITIATED_SHUTDOWN_BEHAVIOUR("instanceInitiatedShutdownBehavior");
        private final String value;

        InstanceAttribute(final String value)
        {
            this.value = value;
        }

        @Override
        public String toString()
        {
            return value;
        }
    }

    /**
     * This enum extends AWS SDK's own enum by adding EBS states and spot request states (we want to treat
     * spot requests like instances)
     */
    public enum InstanceStateName
    {
        //Spot request states
        SpotOpen(SpotInstanceRequestState.OPEN),
        SpotActive(SpotInstanceRequestState.ACTIVE),
        SpotCancelled(SpotInstanceRequestState.CANCELLED),
        SpotClosed(SpotInstanceRequestState.CLOSED),
        SpotFailed(SpotInstanceRequestState.FAILED),

        //instance states
        Pending(com.amazonaws.services.ec2.model.InstanceStateName.Pending),
        Running(com.amazonaws.services.ec2.model.InstanceStateName.Running),
        ShuttingDown(com.amazonaws.services.ec2.model.InstanceStateName.ShuttingDown),
        Terminated(com.amazonaws.services.ec2.model.InstanceStateName.Terminated),

        //EBS states
        Stopping("stopping"),
        Stopped("stopped"),

        //keeping some methods happy
        UNIDENTIFIED("unidentified");
        private final String state;

        InstanceStateName(final String state)
        {
            this.state = state;
        }

        private InstanceStateName(final com.amazonaws.services.ec2.model.InstanceStateName state)
        {
            this.state = state.toString();
        }

        InstanceStateName(SpotInstanceRequestState state)
        {
            this.state = state.stateName;
        }

        @Override
        public String toString()
        {
            return state;
        }

        public static InstanceStateName fromValue(String value)
        {
            Preconditions.checkArgument(StringUtils.isNotEmpty(value), "Value cannot be null or empty!");

            for (InstanceStateName name : values())
            {
                if (name.toString().equals(value))
                {
                    return name;
                }
            }
            throw new IllegalArgumentException("Cannot create enum from " + value + " value!");
        }

        public boolean is(@NotNull InstanceState state) {
            return this.state.equals(state.getName());
        }

        public static InstanceStateName fromValue(final InstanceState state)
        {
            return fromValue(state.getName());
        }
    }

    public enum SpotFleetRequestState
    {
        SUBMITTED("submitted"),
        ACTIVE("active"),
        CANCELLED("cancelled"),
        FAILED("failed"),
        CANCELLED_RUNNING("cancelled_running"),
        CANCELLED_TERMINATING("cancelled_terminating"),
        MODIFYING("modifying");

        private final String stateName;

        SpotFleetRequestState(final String stateName)
        {
            this.stateName = stateName;
        }

        /**
         * @return true if the state won't change interestingly anymore
         */
        public static boolean isFinal(@Nullable final String state)
        {
            return CANCELLED.is(state) || FAILED.is(state) || CANCELLED_RUNNING.is(state) || CANCELLED_TERMINATING.is(state);
        }

        public boolean is(@Nullable final String state)
        {
            return stateName.equals(state);
        }
    }

    public enum SpotInstanceRequestState
    {
        OPEN("open"),
        ACTIVE("active"),
        CANCELLED("cancelled"),
        CLOSED("closed"),
        FAILED("failed");
        private final String stateName;

        SpotInstanceRequestState(final String stateName)
        {
            this.stateName = stateName;
        }

        /**
         * @return true if the state won't change anymore
         */
        public static boolean isFinal(@Nullable final String state)
        {
            return CANCELLED.is(state) || CLOSED.is(state) || FAILED.is(state);
        }

        public boolean is(@Nullable final String state)
        {
            return stateName.equals(state);
        }

        @Override
        public String toString()
        {
            return stateName;
        }
    }

    public static class SpotInstanceRequestType
    {
        public static final String ONE_TIME = "one-time";

        private SpotInstanceRequestType()
        {
        }
    }
}
