package com.atlassian.aws.ec2;

import com.amazonaws.services.ec2.model.InstanceType;
import com.amazonaws.services.ec2.model.SpotPrice;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class SpotPrices {
    private static final Logger log = Logger.getLogger(SpotPrices.class);

    private final Set<String> products = Sets.newTreeSet();

    private final LoadingCache<Pair<InstanceType, String>, Map<String, Price>> spotPricesCache = CacheBuilder.newBuilder().build(new CacheLoader<Pair<InstanceType, String>, Map<String, Price>>() {
        @Override
        public Map<String, Price> load(Pair<InstanceType, String> key) {
            return new HashMap<>();
        }
    });

    public static final class Price implements Comparable<Price> {
        private final Date timestamp;
        private final double spotPrice;

        public Price(final double spotPrice, final Date timestamp) {
            this.timestamp = timestamp;
            this.spotPrice = spotPrice;
        }

        public Date getTimestamp() {
            return timestamp;
        }

        public double getSpotPrice() {
            return spotPrice;
        }

        @Override
        public String toString() {
            return String.valueOf(spotPrice);
        }

        @Override
        public int compareTo(@NotNull Price o) {
            return Double.compare(spotPrice, o.spotPrice);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            Price price = (Price) o;

            if (Double.compare(price.spotPrice, spotPrice) != 0) return false;

            return true;
        }

        @Override
        public int hashCode() {
            long temp = Double.doubleToLongBits(spotPrice);
            return (int) (temp ^ (temp >>> 32));
        }
    }

    public SpotPrices(final List<SpotPrice> spotPrices) {
        spotPrices.forEach(this::add);
    }

    public SpotPrices() {
    }


    @Nullable
    public Price get(final String product, final InstanceType instanceType, final String availabilityZone) {
        final Pair<InstanceType, String> key = makeKey(instanceType, product);
        final Map<String, Price> priceForAz = spotPricesCache.getIfPresent(key);
        if (priceForAz==null) {
            return null;
        }
        return priceForAz.get(availabilityZone);
    }

    @Nullable
    public Pair<Price, Price> get(final String product, final InstanceType instanceType) {
        final Map<String, Price> priceForAz = spotPricesCache.getIfPresent(makeKey(instanceType, product));
        if (priceForAz==null) {
            return null;
        }
        return Pair.of(Collections.min(priceForAz.values()), Collections.max(priceForAz.values()));
    }

    @NotNull
    public Set<String> getProducts() {
        return products;
    }

    @Override
    public String toString() {
        return spotPricesCache.asMap().toString();
    }

    private void add(final SpotPrice spotPrice) {
        final InstanceType instanceType;
        try {
            instanceType = InstanceType.fromValue(spotPrice.getInstanceType());
        }catch (final IllegalArgumentException ignored) {
            log.debug("Skipping spot price for " + spotPrice.getInstanceType());
            return;
        }

        final String product = spotPrice.getProductDescription();
        final Pair<InstanceType, String> key = makeKey(instanceType, product);
        final Map<String, Price> pricesForAllZones = spotPricesCache.getUnchecked(key);

        products.add(spotPrice.getProductDescription());
        pricesForAllZones.put(spotPrice.getAvailabilityZone(), new Price(Double.valueOf(spotPrice.getSpotPrice()), spotPrice.getTimestamp()));
    }

    private static Pair<InstanceType, String> makeKey(final InstanceType instanceType, final String product) {
        return Pair.of(instanceType, product);
    }
}
