package com.atlassian.aws.ec2;

import com.amazonaws.services.ec2.AmazonEC2;
import com.amazonaws.services.ec2.model.AuthorizeSecurityGroupIngressRequest;
import com.amazonaws.services.ec2.model.IpPermission;
import com.amazonaws.services.ec2.model.IpRange;
import com.amazonaws.services.ec2.model.Ipv6Range;
import com.amazonaws.services.ec2.model.SecurityGroup;
import org.jetbrains.annotations.NotNull;

public final class AmazonEc2Utils {

    private AmazonEc2Utils() {
    }

    public static void ensureInboundTrafficIsAllowed(final AmazonEC2 amazonEc2, @NotNull final SecurityGroup group, @NotNull final Protocol protocol, @NotNull final String cidrIpRange, final int port) {
        final boolean permissionAlreadyExists = !EC2Utils.getMatchingIpPermissions(group, protocol, cidrIpRange, port).isEmpty();
        if (permissionAlreadyExists) {
            return;
        }

        final IpPermission ipPermission = new IpPermission()
                .withIpProtocol(protocol.getValue())
                .withFromPort(port)
                .withToPort(port);

        if (isIpv6Range(cidrIpRange)) {
            ipPermission
                    .withIpv6Ranges(
                            new Ipv6Range().withCidrIpv6(cidrIpRange)
                    );
        } else {
            ipPermission
                    .withIpv4Ranges(
                            new IpRange().withCidrIp(cidrIpRange)
                    );
        }

        final AuthorizeSecurityGroupIngressRequest authorizeSecurityGroupIngressRequest =
                new AuthorizeSecurityGroupIngressRequest()
                        .withGroupId(group.getGroupId())
                        .withIpPermissions(ipPermission);
        amazonEc2.authorizeSecurityGroupIngress(authorizeSecurityGroupIngressRequest);
    }

    private static boolean isIpv6Range(final String cidrIpRange) {
        return cidrIpRange.contains(":");
    }
}
