package com.atlassian.aws.utils;

import com.amazonaws.annotation.NotThreadSafe;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Iterator;
import java.util.Set;

public class FullListObjectsResult implements Iterable<S3ObjectSummary>
{
    private final AmazonS3Client s3Client;
    private final String bucketName;
    private final String s3PathForArtifact;

    /**
     * @deprecated since 5.6 use the static factory methods instead
     */
    @Deprecated
    public FullListObjectsResult(final AmazonS3Client s3Client, final String bucketName, final String s3PathForArtifact)
    {
        this.s3Client = s3Client;
        this.bucketName = bucketName;
        this.s3PathForArtifact = s3PathForArtifact;
    }

    @Override
    @Deprecated
    public Iterator<S3ObjectSummary> iterator()
    {
        final ObjectListing firstObjectListing = s3Client.listObjects(bucketName, s3PathForArtifact);

        return iterator(s3Client, firstObjectListing, null);
    }

    private static Iterator<S3ObjectSummary> iterator(final AmazonS3Client s3Client, final ObjectListing firstObjectListing, @Nullable final ObjectListingChangeCallback objectListingCallable)
    {
        return new Iterator<S3ObjectSummary>()
        {
            private ObjectListing currentObjectListing;
            private Iterator<S3ObjectSummary> objectListingIterator;

            {
                setCurrentObjectListing(firstObjectListing);
            }

            @Override
            public boolean hasNext()
            {
                return currentObjectListing.isTruncated() || objectListingIterator.hasNext();
            }

            @Override
            public S3ObjectSummary next()
            {
                if (!objectListingIterator.hasNext() && currentObjectListing.isTruncated())
                {
                    setCurrentObjectListing(s3Client.listNextBatchOfObjects(currentObjectListing));
                }

                return objectListingIterator.next();
            }

            @Override
            public void remove()
            {
                objectListingIterator.remove();
            }

            private void setCurrentObjectListing(final ObjectListing objectListing)
            {
                currentObjectListing = objectListing;
                objectListingIterator = currentObjectListing.getObjectSummaries().iterator();
                if (objectListingCallable!=null)
                {
                    objectListingCallable.onNewObjectListing(objectListing);
                }
            }
        };
    }

    public static ContinuousObjectListing listObjects(@NotNull final AmazonS3Client s3Client, final String bucketName, final String s3PathForArtifact, @Nullable final ObjectListingChangeCallback objectListingCallable)
    {
        return new ContinuousObjectListing(s3Client, new ListObjectsRequest().withBucketName(bucketName).withPrefix(s3PathForArtifact));
    }

    public static ContinuousObjectListing listObjects(final AmazonS3Client s3Client, final ListObjectsRequest listObjectsRequest)
    {
        return new ContinuousObjectListing(s3Client, listObjectsRequest);
    }

    public static interface ObjectListingChangeCallback
    {
        void onNewObjectListing(final ObjectListing objectListing);
    }

    @NotThreadSafe
    public static class ContinuousObjectListing
    {
        private final AmazonS3Client s3Client;
        private final ListObjectsRequest listObjectsRequest;
        private volatile Set<String> commonPrefixes;
        private final ImmutableSet.Builder<String> truncatedCommonPrefixes = ImmutableSet.builder();

        public ContinuousObjectListing(final AmazonS3Client s3Client, final ListObjectsRequest listObjectsRequest)
        {
            this.s3Client = s3Client;
            this.listObjectsRequest = listObjectsRequest;
        }

        public Iterable<S3ObjectSummary> getS3ObjectSummaries()
        {
            final ObjectListing firstObjectListing = s3Client.listObjects(listObjectsRequest);

            final ObjectListingChangeCallback objectListingCallable = new ObjectListingChangeCallback()
            {
                @Override
                public void onNewObjectListing(final ObjectListing objectListing)
                {
                    truncatedCommonPrefixes.addAll(objectListing.getCommonPrefixes());
                    if (!objectListing.isTruncated())
                    {
                        commonPrefixes = truncatedCommonPrefixes.build();
                    }
                }
            };
            return new Iterable<S3ObjectSummary>()
            {
                @Override
                public Iterator<S3ObjectSummary> iterator()
                {
                    return FullListObjectsResult.iterator(s3Client, firstObjectListing, objectListingCallable);
                }
            };
        }

        /**
         * Note: if you're going to iterate through all {@link #getS3ObjectSummaries()}, do it before calling this method.
         * @return
         */
        public Iterable<String> getCommonPrefixes()
        {
            return new Iterable<String>()
            {
                @Override
                public Iterator<String> iterator()
                {
                    return lazyCommonPrefixesIterator();
                }
            };
        }

        private Iterator<String> lazyCommonPrefixesIterator()
        {
            return new Iterator<String>() {

                private final Supplier<Iterator<String>> iterator = Suppliers.memoize(new Supplier<Iterator<String>>()
                {
                    @Override
                    public Iterator<String> get()
                    {
                        if (commonPrefixes == null)
                        {
                            Iterables.size(getS3ObjectSummaries());
                        }
                        return commonPrefixes.iterator();
                    }
                });

                @Override
                public boolean hasNext()
                {
                    return iterator.get().hasNext();
                }

                @Override
                public String next()
                {
                    return iterator.get().next();
                }

                @Override
                public void remove()
                {
                    throw new UnsupportedOperationException();
                }
            };
        }
    }
}