package com.atlassian.aws.ec2.model;

import com.amazonaws.services.ec2.model.Subnet;
import com.google.common.base.Preconditions;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.List;
import java.util.function.Function;

/**
 * Encapsulates VPC subnet string id.
 */
public final class SubnetId extends ResourceId<SubnetId>
{
    public static final SubnetId NO_SUBNET = new SubnetId(false);

    public static final Function<Subnet, String> GET_AZ = Subnet::getAvailabilityZone;

    public SubnetId(@NotNull final String subnetId)
    {
        super(subnetId);
        Preconditions.checkArgument(subnetId.startsWith("subnet-"));
    }

    private SubnetId(final boolean b)
    {
        super("");
    }

    public boolean isUndefined()
    {
        //noinspection ObjectEquality
        return this == NO_SUBNET;
    }

    @NotNull
    public static SubnetId from(@NotNull final String subnetId)
    {
        return new SubnetId(subnetId);
    }

    @Nullable
    public static SubnetId fromNullable(@Nullable final String subnetId)
    {
        return subnetId == null ? null : from(subnetId);
    }

    @NotNull
    public static SubnetId from(@NotNull final Subnet subnet)
    {
        return from(subnet.getSubnetId());
    }

    @NotNull
    public static List<SubnetId> from(final Collection<String> subnetIds)
    {
        return ResourceId.fromCollection(subnetIds, SubnetId::from);
    }
}