package com.atlassian.aws.ec2.model;

import com.amazonaws.services.ec2.model.GroupIdentifier;
import com.amazonaws.services.ec2.model.SecurityGroup;
import com.google.common.base.Preconditions;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

/**
 * Encapsulates Security group string id.
 */
public class SecurityGroupId extends ResourceId<SecurityGroupId>
{

    private static final Function<SecurityGroupId, GroupIdentifier> TO_GROUP_IDENTIFIER = securityGroupId -> new GroupIdentifier()
            .withGroupId(securityGroupId.getId());

    public SecurityGroupId(@NotNull final String id)
    {
        super(id);
        Preconditions.checkArgument(id.startsWith("sg-"));
    }

    public static SecurityGroupId from(@NotNull final String id)
    {
        return new SecurityGroupId(id);
    }

    public static SecurityGroupId from(@NotNull final SecurityGroup securityGroup)
    {
        return from(securityGroup.getGroupId());
    }

    public static Collection<GroupIdentifier> asGroupIdentifiers(final Iterable<SecurityGroupId> securityGroups)
    {
        return StreamSupport.stream(securityGroups.spliterator(), false)
                .map(TO_GROUP_IDENTIFIER)
                .collect(Collectors.toList());
    }
}
