package com.atlassian.aws.ec2.configuration;

import com.atlassian.aws.ec2.awssdk.AwsSupportConstants;
import com.atlassian.aws.ec2.awssdk.RegionSupport;
import com.google.common.base.Joiner;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

public class ImageFileDao
{
    private static final String PROP_AGENT_DATA_BUCKET_STEM = "agent.data.bucket";
    private static final String PROP_SERVER_VERSION_SPECIFIC_KEY = "server.version.key";
    private static final String PROP_BOOTSTRAP_KEY = "bootstrap.key";
    private static final String PROP_OLD_BOOTSTRAP_KEY = "assembly.key"; //can be removed around 5.11
    private static final String PROP_OLD_AGENT_DATA_BUCKET_STEM = "assembly.bucket"; //can be removed around 5.11
    private static final String IMAGE_PROPERTY_PREFIX = "image";

    private ImageFileDao()
    {
    }

    public static String getPropertyName(Object...args)
    {
        return IMAGE_PROPERTY_PREFIX + "." + Joiner.on(".").join(args);
    }

    @NotNull
    public static ImageData readOrCreate(final File amiIDFile) throws IOException
    {
        if (amiIDFile.canRead())
        {
            return read(amiIDFile);
        }
        else
        {
            return read((InputStream) null);
        }
    }

    @NotNull
    public static ImageData read(final File amiIDFile) throws IOException
    {
        FileInputStream fis = new FileInputStream(amiIDFile);
        try
        {
            return read(fis);
        }
        finally
        {
            fis.close();
        }
    }
    
    public static ImageData read(@Nullable final InputStream fis) throws IOException
    {
        ImageData allImageData = new ImageData();

        if (fis==null)
        {
            return allImageData;
        }
        
        Properties props = new Properties();
        props.load(fis);

        allImageData.setAssemblyBucketStem(props.getProperty(PROP_AGENT_DATA_BUCKET_STEM, props.getProperty(PROP_OLD_AGENT_DATA_BUCKET_STEM)));
        allImageData.setBootstrapKey(props.getProperty(PROP_BOOTSTRAP_KEY, props.getProperty(PROP_OLD_BOOTSTRAP_KEY)));
        allImageData.setServerVersionSpecificDataKey(props.getProperty(PROP_SERVER_VERSION_SPECIFIC_KEY));

        Set<Map.Entry<String,String>> entries = (Set)props.entrySet();
        for (Map.Entry<String, String> keyValue : entries)
        {
            final String key = keyValue.getKey();
            if (!key.startsWith(IMAGE_PROPERTY_PREFIX))
            {
                continue;
            }
            final String[] split = StringUtils.split(key, ".", 7);
            final AwsSupportConstants.Region region = AwsSupportConstants.Region.valueOf(split[1]);
            final AwsSupportConstants.RootDeviceType rootDeviceType = AwsSupportConstants.RootDeviceType.valueOf(split[2]);
            final AwsSupportConstants.Architecture architecture = AwsSupportConstants.Architecture.valueOf(split[3]);
            final AwsSupportConstants.Platform platform = AwsSupportConstants.Platform.valueOf(split[4]);

            final AwsSupportConstants.Virtualisation virtualisation;
            final String osName;
            if (split.length == 7)
            {
                virtualisation = AwsSupportConstants.Virtualisation.valueOf(StringUtils.upperCase(split[5]));
                osName = split[6];
            }
            else
            {
                virtualisation = AwsSupportConstants.Virtualisation.UNKNOWN;
                osName = split[5];
            }

            allImageData.add(new ImageData.ImageDefinition(RegionSupport.toAwsSdkRegion(region), rootDeviceType, architecture, platform, virtualisation, osName, keyValue.getValue()));
        }

        return allImageData;
    }

    public static ImageData readArtifact(@NotNull final String artifactId) throws IOException
    {
        final InputStream inputStream = ImageFileDao.class.getResourceAsStream("/" + artifactId + ".aws");
        if (inputStream == null)
        {
            throw new FileNotFoundException();
        }

        try
        {
            ImageData data = ImageFileDao.read(inputStream);
            return data;
        }
        finally
        {
            inputStream.close();
        }
    }

    public static void write(final ImageData imageData, final File amiIDFile) throws IOException
    {
        File parent = amiIDFile.getParentFile();
        if (parent!=null)
        {
            parent.mkdirs();
        }

        if (!amiIDFile.exists())
        {
            amiIDFile.createNewFile();
        }

        Properties props = new Properties();


        putIfNotNull(props, PROP_AGENT_DATA_BUCKET_STEM, imageData.getAssemblyBucketStem());
        putIfNotNull(props, PROP_SERVER_VERSION_SPECIFIC_KEY, imageData.getServerVersionSpecificDataKey());
        putIfNotNull(props, PROP_BOOTSTRAP_KEY, imageData.getBootstrapKey());

        for (ImageData.ImageDefinition imageDefinition : imageData.getImageDefinitions())
        {
            props.put(
                    getPropertyName(
                            imageDefinition.getRegion(),
                            imageDefinition.getRootDeviceType(),
                            imageDefinition.getImageArchitecture(),
                            imageDefinition.getImagePlatform(),
                            imageDefinition.getVirtualisation(),
                            imageDefinition.getOsName()
                    ), imageDefinition.getImageId());
        }

        final FileOutputStream out = new FileOutputStream(amiIDFile);
        try
        {
            props.store(out, null);
        }
        finally
        {
            out.close();
        }
    }

    private static void putIfNotNull(@NotNull final Properties properties, @NotNull final String key, @Nullable final String value)
    {
        if (value != null)
        {
            properties.put(key, value);
        }
    }
}
