package com.atlassian.aws.ec2.configuration;

import com.amazonaws.regions.Region;
import com.amazonaws.regions.Regions;
import com.atlassian.aws.ec2.awssdk.AwsSupportConstants;
import com.atlassian.aws.ec2.awssdk.RegionSupport;
import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.List;

import static com.atlassian.aws.ec2.awssdk.RegionSupport.toAtlassianAwsRegion;

public class ImageData
{
    private final List<ImageDefinition> imageDefinitions = new ArrayList<>();
    private String assemblyBucketStem;
    private String serverVersionSpecificDataKey;
    private String bootstrapKey;

    public static class ImageDefinition
    {
        private final @NotNull Regions region;
        private final AwsSupportConstants.RootDeviceType rootDeviceType;
        private final AwsSupportConstants.Architecture imageArchitecture;
        private final AwsSupportConstants.Platform imagePlatform;
        private final AwsSupportConstants.Virtualisation virtualisation;
        private final String osName;
        private String imageId;

        public ImageDefinition(@NotNull final Regions region,
                               @NotNull final AwsSupportConstants.RootDeviceType rootDeviceType,
                               @NotNull final AwsSupportConstants.Architecture imageArchitecture,
                               @NotNull final AwsSupportConstants.Platform imagePlatform,
                               @NotNull final AwsSupportConstants.Virtualisation virtualisation,
                               @NotNull final String osName,
                               @NotNull final String imageId)
        {
            this.region = region;
            this.rootDeviceType = rootDeviceType;
            this.imageArchitecture = imageArchitecture;
            this.imagePlatform = imagePlatform;
            this.virtualisation = virtualisation;
            this.osName = osName;
            this.imageId = imageId;
        }

        public String getImageId()
        {
            return imageId;
        }

        public void setImageId(final String value)
        {
            imageId = value;
        }

        public AwsSupportConstants.Region getRegion()
        {
            return toAtlassianAwsRegion(region);
        }

        public @NotNull Regions getAwsSdkRegion()
        {
            return region;
        }
        
        public AwsSupportConstants.RootDeviceType getRootDeviceType()
        {
            return rootDeviceType;
        }

        public AwsSupportConstants.Architecture getImageArchitecture()
        {
            return imageArchitecture;
        }

        public AwsSupportConstants.Platform getImagePlatform()
        {
            return imagePlatform;
        }

        public AwsSupportConstants.Virtualisation getVirtualisation()
        {
            return virtualisation;
        }

        public String getOsName()
        {
            return osName;
        }
    }

    public String getAssemblyBucketStem()
    {
        return assemblyBucketStem;
    }

    public String getAssemblyBucket(final AwsSupportConstants.Region region)
    {
        return getNameForRegion(assemblyBucketStem, region);
    }

    public void setAssemblyBucketStem(final String assemblyBucketStem)
    {
        this.assemblyBucketStem = assemblyBucketStem;
    }

    public String getServerVersionSpecificDataKey()
    {
        return serverVersionSpecificDataKey;
    }

    public void setServerVersionSpecificDataKey(final String serverVersionSpecificDataKey)
    {
        this.serverVersionSpecificDataKey = serverVersionSpecificDataKey;
    }

    public String getBootstrapKey()
    {
        return bootstrapKey;
    }

    public void setBootstrapKey(final String bootstrapKey)
    {
        this.bootstrapKey = bootstrapKey;
    }

    public ImageData add(final ImageDefinition imageDefinition)
    {
        imageDefinitions.add(imageDefinition);
        return this;
    }

    @NotNull
    public List<ImageDefinition> getImageDefinitions()
    {
        return imageDefinitions;
    }

    @NotNull
    public static String getNameForRegion(@NotNull final String nameStem, @NotNull AwsSupportConstants.Region region)
    {
        return nameStem + "-" + region.getNormalisedName();
    }

    @NotNull
    public static String getNameForAwsRegion(@NotNull final String nameStem, @NotNull Region region)
    {
        return nameStem + "-" + RegionSupport.getNormalisedName(region);
    }
}
