package com.atlassian.aws.ec2.caches;

import com.amazonaws.services.ec2.AmazonEC2;
import com.amazonaws.services.ec2.AmazonEC2Async;
import com.amazonaws.services.ec2.AmazonEC2Client;
import com.amazonaws.services.ec2.model.DescribeSpotInstanceRequestsRequest;
import com.amazonaws.services.ec2.model.SpotInstanceRequest;
import com.atlassian.aws.AmazonServiceErrorCode;
import com.atlassian.aws.AwsOmeCache;
import org.apache.log4j.Logger;

import java.time.Duration;
import java.util.List;

/**
 * This class provides access to cached spot request data and makes sure that the cached data is recent enough.
 * Duration after which the data is considered to be stale is configurable via constructor paramaters.
 *
 */
public class SpotRequestCache extends AwsOmeCache<SpotInstanceRequest>
{
    private static final Logger log = Logger.getLogger(SpotRequestCache.class);
    private final AmazonEC2 ec2Client;

    /**
     * @param maximumDataAge the maximum age after which the data has to be refreshed from EC2 server
     */
    public SpotRequestCache(final AmazonEC2Client ec2Client, Duration maximumDataAge)
    {
        super(log, maximumDataAge);
        
        this.ec2Client = ec2Client;
    }

    /**
     * @param ec2Client
     */
    public SpotRequestCache(final AmazonEC2Async ec2Client)
    {
        super(log);
        this.ec2Client = ec2Client;
    }

    @Override
    protected List<SpotInstanceRequest> refreshAllData()
    {
        DescribeSpotInstanceRequestsRequest describeSpotRequest = new DescribeSpotInstanceRequestsRequest();

        return ec2Client.describeSpotInstanceRequests(describeSpotRequest).getSpotInstanceRequests();
    }

    @Override
    protected void onResourceLookupFailure(final String resourceId)
    {
        throw new Ec2CacheMissException(resourceId, AmazonServiceErrorCode.INVALID_SPOT_INSTANCE_REQUEST_ID_NOT_FOUND);
    }

    @Override
    protected String toResourceId(final SpotInstanceRequest resource)
    {
        return resource.getSpotInstanceRequestId();
    }
}
