package com.atlassian.aws.ec2.awssdk.launch;

import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.services.ec2.model.LaunchSpecification;
import com.amazonaws.services.ec2.model.RequestSpotInstancesRequest;
import com.amazonaws.services.ec2.model.SpotInstanceRequest;
import com.amazonaws.services.ec2.model.SpotPlacement;
import com.atlassian.aws.AWSAccount;
import com.atlassian.aws.AmazonServiceErrorCode;
import com.atlassian.aws.ec2.EC2InstanceType;
import com.atlassian.aws.ec2.EC2Utils;
import com.atlassian.aws.ec2.InstanceLaunchConfiguration;
import com.atlassian.aws.ec2.InstancePaymentType;
import com.atlassian.aws.ec2.InstanceReservationDescription;
import com.atlassian.aws.ec2.InstanceStatus;
import com.atlassian.aws.ec2.SubnetChooser;
import com.atlassian.aws.ec2.awssdk.AwsSpotInstanceReservationDescription;
import com.atlassian.aws.ec2.awssdk.AwsSupportConstants;
import com.atlassian.aws.ec2.awssdk.InstanceLauncher;
import com.atlassian.aws.ec2.model.SecurityGroupId;
import com.atlassian.aws.ec2.model.SubnetId;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.log4j.Logger;
import org.jetbrains.annotations.NotNull;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import static com.atlassian.aws.ec2.awssdk.launch.LauncherUtils.*;
import static com.atlassian.aws.ec2.model.SecurityGroupId.asGroupIdentifiers;

class AwsSpotInstanceLauncher implements InstanceLauncher
{
    private static final Logger log = Logger.getLogger(AwsSpotInstanceLauncher.class);

    private final AWSAccount awsAccount;
    private final SubnetChooser subnetChooser;
    private final InstanceLaunchConfiguration instanceConfiguration;
    private final InstanceStatus instanceStatus;
    private final InstanceLauncher fallbackLauncher;

    AwsSpotInstanceLauncher(final AWSAccount awsAccount, final SubnetChooser subnetChooser, final InstanceLaunchConfiguration instanceConfiguration, final InstanceStatus instanceStatus, InstanceLauncher fallbackLauncher)
    {
        this.awsAccount = awsAccount;
        this.subnetChooser = subnetChooser;
        this.instanceConfiguration = instanceConfiguration;
        this.instanceStatus = instanceStatus;
        this.fallbackLauncher = fallbackLauncher;
    }

    @Override
    public Collection<InstanceReservationDescription> call() throws IOException, AmazonClientException
    {
        final int spotRequestTimeoutSeconds = instanceConfiguration.getSpotRequestTimeoutSeconds();
        final Date spotRequestValidityEnd = DateUtils.addSeconds(new Date(), spotRequestTimeoutSeconds);
        instanceStatus.setDeadline(2 * spotRequestTimeoutSeconds);

        final LaunchSpecification spotLaunchSpecification = createSpotLaunchSpecification(instanceConfiguration);

        final RequestSpotInstancesRequest requestSpotInstance = new RequestSpotInstancesRequest()
                .withSpotPrice(Double.toString(instanceConfiguration.getSpotInstanceBid()))
                .withInstanceCount(1)
                .withValidUntil(spotRequestValidityEnd)
                .withType(AwsSupportConstants.SpotInstanceRequestType.ONE_TIME)
                .withLaunchSpecification(spotLaunchSpecification);

        List<SpotInstanceRequest> placedSpotInstanceRequest = Collections.emptyList();
        try
        {
            placedSpotInstanceRequest = awsAccount.getAmazonEc2().requestSpotInstances(requestSpotInstance).getSpotInstanceRequests();
        }
        catch (final AmazonServiceException e)
        {
            if (!AmazonServiceErrorCode.MAX_SPOT_INSTANCE_COUNT_EXCEEDED.is(e))
            {
                throw e;
            }
        }

        if (placedSpotInstanceRequest.isEmpty())
        {
            log.warn("Unable to place a spot instance request, proceeding with regular instance order");
            return fallbackLauncher.call();
        }
        instanceStatus.setInstancePaymentType(InstancePaymentType.SPOT);

        if (placedSpotInstanceRequest.size() > 1)
        {
            log.warn("A request for a single instance resulted in " + placedSpotInstanceRequest.size() + " being placed. Ignoring spurious spot instance requests.");
        }

        instanceStatus.setSpotInstanceRequestId(Iterables.getOnlyElement(placedSpotInstanceRequest).getSpotInstanceRequestId());

        return AwsSpotInstanceReservationDescription.create(placedSpotInstanceRequest);
    }

    //Spot launch specification
    private LaunchSpecification createSpotLaunchSpecification(final InstanceLaunchConfiguration instanceConfiguration)
    {
        final EC2InstanceType instanceType = getInstanceType(instanceConfiguration);
        final String imageId = instanceConfiguration.getImage().getId();
        final LaunchSpecification launchSpecification = new LaunchSpecification()
                .withImageId(imageId)
                .withKeyName(instanceConfiguration.getKeyName())
                .withIamInstanceProfile(instanceConfiguration.getIamInstanceProfile())
                .withBlockDeviceMappings(getBlockDeviceMappings(awsAccount, imageId, instanceType, instanceConfiguration.getEbsSnapshotId()))
                .withEbsOptimized(instanceConfiguration.isEbsOptimised())
                .withInstanceType(instanceType.getAwsInstanceType());

        if (isVpcEnabled(instanceConfiguration))
        {
            final SubnetId subnetId = subnetChooser.choose(instanceConfiguration.getSubnets(), instanceType);
            Preconditions.checkNotNull(subnetId);
            final Iterable<SecurityGroupId> securityGroupIds = instanceConfiguration.getSecurityGroups(subnetId);

            if (instanceConfiguration.shouldAssociatePublicIp())
            {
                launchSpecification.withNetworkInterfaces(newPublicIpForVpc(subnetId, securityGroupIds));
            }
            else
            {
                launchSpecification
                        .withSubnetId(subnetId.getId())
                        .withAllSecurityGroups(asGroupIdentifiers(securityGroupIds));
            }
        }
        else
        {
            launchSpecification
                    .withPlacement(new SpotPlacement(instanceConfiguration.getRequestedAvailabilityZone()))
                    .withAllSecurityGroups(asGroupIdentifiers(instanceConfiguration.getSecurityGroups(null)));
        }

        launchSpecification.setUserData(EC2Utils.getUserDataAsString(instanceConfiguration.getUserData()));

        return launchSpecification;
    }

    @NotNull
    private EC2InstanceType getInstanceType(final InstanceLaunchConfiguration instanceConfiguration)
    {
        final Iterable<EC2InstanceType> instanceTypes = instanceConfiguration.getInstanceTypes();
        final EC2InstanceType instanceType = Iterables.get(instanceTypes, 0);
        if (Iterables.size(instanceTypes)>1)
        {
            log.info("Launching a non-spot fleet instance and the image configuration has more than one instance type defined, using " + instanceType);
        }
        return instanceType;
    }
}