package com.atlassian.aws.ec2.awssdk.launch;

import com.amazonaws.services.ec2.AmazonEC2Async;
import com.amazonaws.services.ec2.model.RequestSpotFleetRequest;
import com.amazonaws.services.ec2.model.RequestSpotFleetResult;
import com.amazonaws.services.ec2.model.SpotFleetLaunchSpecification;
import com.amazonaws.services.ec2.model.SpotFleetRequestConfigData;
import com.amazonaws.services.ec2.model.SpotPlacement;
import com.atlassian.aws.AWSAccount;
import com.atlassian.aws.ec2.EC2InstanceType;
import com.atlassian.aws.ec2.EC2Utils;
import com.atlassian.aws.ec2.InstanceLaunchConfiguration;
import com.atlassian.aws.ec2.InstancePaymentType;
import com.atlassian.aws.ec2.InstanceReservationDescription;
import com.atlassian.aws.ec2.InstanceStatus;
import com.atlassian.aws.ec2.SubnetChooser;
import com.atlassian.aws.ec2.awssdk.AwsSpotFleetInstanceRequest;
import com.atlassian.aws.ec2.awssdk.InstanceLauncher;
import com.atlassian.aws.ec2.model.SecurityGroupId;
import com.atlassian.aws.ec2.model.SubnetId;
import com.google.common.base.Preconditions;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.time.DateUtils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;

import static com.atlassian.aws.ec2.awssdk.launch.LauncherUtils.getBlockDeviceMappings;
import static com.atlassian.aws.ec2.awssdk.launch.LauncherUtils.isVpcEnabled;
import static com.atlassian.aws.ec2.awssdk.launch.LauncherUtils.newPublicIpForVpc;
import static com.atlassian.aws.ec2.model.SecurityGroupId.asGroupIdentifiers;

public class AwsSpotFleetLauncher implements InstanceLauncher
{
    private final AWSAccount awsAccount;
    private final SubnetChooser subnetChooser;
    private final InstanceLaunchConfiguration instanceConfiguration;
    private final InstanceStatus instanceStatus;
    //private final InstanceLauncher fallbackLauncher;

    AwsSpotFleetLauncher(final AWSAccount awsAccount, final SubnetChooser subnetChooser, final InstanceLaunchConfiguration instanceConfiguration, final InstanceStatus instanceStatus, InstanceLauncher fallbackLauncher)
    {
        this.awsAccount = awsAccount;
        this.subnetChooser = subnetChooser;
        this.instanceConfiguration = instanceConfiguration;
        this.instanceStatus = instanceStatus;
        //this.fallbackLauncher = fallbackLauncher;
    }

    @Override
    public Collection<InstanceReservationDescription> call() throws IOException
    {
        final int spotRequestTimeoutSeconds = instanceConfiguration.getSpotRequestTimeoutSeconds();
        final Date spotRequestValidityEnd = DateUtils.addSeconds(new Date(), spotRequestTimeoutSeconds);
        instanceStatus.setDeadline(2 * spotRequestTimeoutSeconds);

        final AmazonEC2Async amazonEc2 = awsAccount.getAmazonEc2();

        final Collection<SpotFleetLaunchSpecification> launchSpecifications = createLaunchSpecifications();

        final SpotFleetRequestConfigData spotFleetRequestConfigData =
                new SpotFleetRequestConfigData()
                        .withSpotPrice(Double.toString(instanceConfiguration.getSpotInstanceBid()))
                        .withTargetCapacity(1)
                        .withIamFleetRole("arn:aws:iam::121852097033:role/PrzemekFleetRole2") //todo
                        .withValidUntil(spotRequestValidityEnd)
                        .withLaunchSpecifications(launchSpecifications)
                        .withTerminateInstancesWithExpiration(false)
                        .withClientToken(RandomStringUtils.randomAlphanumeric(64));

        final RequestSpotFleetRequest requestSpotFleetRequest =
                new RequestSpotFleetRequest()
                        .withSpotFleetRequestConfig(spotFleetRequestConfigData);
        final RequestSpotFleetResult requestSpotFleetResult = amazonEc2.requestSpotFleet(requestSpotFleetRequest);

        instanceStatus.setInstancePaymentType(InstancePaymentType.SPOT);
        instanceStatus.setSpotInstanceRequestId(requestSpotFleetResult.getSpotFleetRequestId());

        return AwsSpotFleetInstanceRequest.create(requestSpotFleetResult, launchSpecifications);
    }

    private Collection<SpotFleetLaunchSpecification> createLaunchSpecifications()
    {
        final Iterable<EC2InstanceType> instanceTypes = instanceConfiguration.getInstanceTypes();
        final String imageId = instanceConfiguration.getImage().getId();

        final Collection<SpotFleetLaunchSpecification> specifications = new ArrayList<>();
        for (final EC2InstanceType instanceType : instanceTypes)
        {
            final SpotFleetLaunchSpecification launchSpecification = new SpotFleetLaunchSpecification()
                    .withImageId(imageId)
                    .withKeyName(instanceConfiguration.getKeyName())
                    .withIamInstanceProfile(instanceConfiguration.getIamInstanceProfile())
                    .withBlockDeviceMappings(getBlockDeviceMappings(awsAccount, imageId, instanceType, instanceConfiguration.getEbsSnapshotId()))
                    .withEbsOptimized(instanceConfiguration.isEbsOptimised())
                    .withInstanceType(instanceType.getAwsInstanceType());

            if (isVpcEnabled(instanceConfiguration))
            {
                final SubnetId subnetId = subnetChooser.choose(instanceConfiguration.getSubnets(), instanceType);
                Preconditions.checkNotNull(subnetId);
                final Iterable<SecurityGroupId> securityGroupIds = instanceConfiguration.getSecurityGroups(subnetId);

                if (instanceConfiguration.shouldAssociatePublicIp())
                {
                    launchSpecification.withNetworkInterfaces(newPublicIpForVpc(subnetId, securityGroupIds));
                }
                else
                {
                    launchSpecification
                            .withSubnetId(subnetId.getId())
                            .withSecurityGroups(asGroupIdentifiers(securityGroupIds));
                }
            }
            else
            {
                launchSpecification
                        .withPlacement(new SpotPlacement(instanceConfiguration.getRequestedAvailabilityZone()))
                        .withSecurityGroups(asGroupIdentifiers(instanceConfiguration.getSecurityGroups(null)));
            }

            final Object userData = instanceConfiguration.getUserData();
            launchSpecification.setUserData(EC2Utils.getUserDataAsString(userData));

            specifications.add(launchSpecification);
        }

        return specifications;
    }
}
