package com.atlassian.aws.ec2.awssdk.launch;

import com.amazonaws.AmazonServiceException;
import com.amazonaws.services.ec2.model.Placement;
import com.amazonaws.services.ec2.model.RunInstancesRequest;
import com.amazonaws.services.ec2.model.RunInstancesResult;
import com.atlassian.aws.AWSAccount;
import com.atlassian.aws.AmazonServiceErrorCode;
import com.atlassian.aws.ec2.EC2InstanceType;
import com.atlassian.aws.ec2.EC2Utils;
import com.atlassian.aws.ec2.InstanceLaunchConfiguration;
import com.atlassian.aws.ec2.InstancePaymentType;
import com.atlassian.aws.ec2.InstanceReservationDescription;
import com.atlassian.aws.ec2.InstanceStatus;
import com.atlassian.aws.ec2.SubnetChooser;
import com.atlassian.aws.ec2.awssdk.AwsInstanceReservationDescription;
import com.atlassian.aws.ec2.awssdk.AwsSupportConstants;
import com.atlassian.aws.ec2.awssdk.InstanceLauncher;
import com.atlassian.aws.ec2.model.ResourceId;
import com.atlassian.aws.ec2.model.SecurityGroupId;
import com.atlassian.aws.ec2.model.SubnetId;
import com.atlassian.aws.utils.Eithers;
import com.google.common.collect.Iterables;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.util.Collection;

import static com.atlassian.aws.ec2.awssdk.launch.LauncherUtils.getBlockDeviceMappings;
import static com.atlassian.aws.ec2.awssdk.launch.LauncherUtils.isVpcEnabled;
import static com.atlassian.aws.ec2.awssdk.launch.LauncherUtils.newPublicIpForVpc;

class AwsInstanceLauncher implements InstanceLauncher
{
    private static final Logger log = Logger.getLogger(AwsInstanceLauncher.class);

    private final AWSAccount awsAccount;
    private final SubnetChooser subnetChooser;
    private final InstanceLaunchConfiguration instanceConfiguration;
    private final InstanceStatus instanceStatus;

    AwsInstanceLauncher(final AWSAccount awsAccount, final SubnetChooser subnetChooser, final InstanceLaunchConfiguration instanceConfiguration, final InstanceStatus instanceStatus)
    {
        this.awsAccount = awsAccount;
        this.subnetChooser = subnetChooser;
        this.instanceConfiguration = instanceConfiguration;
        this.instanceStatus = instanceStatus;
    }

    @Override
    public Collection<InstanceReservationDescription> call() throws IOException
    {
        instanceStatus.setInstancePaymentType(InstancePaymentType.REGULAR);
        instanceStatus.setDeadline(instanceConfiguration.getStartupTimeoutInSeconds());

        final EC2InstanceType instanceType = getInstanceType(instanceConfiguration);

        Pair<AmazonServiceException, RunInstancesResult> runInstancesResult;
        SubnetId subnetId = subnetChooser.choose(instanceConfiguration.getSubnets(), instanceType);
        final SubnetId firstSubnetId = subnetId;
        do
        {
            runInstancesResult = runInstance(subnetId, instanceType);
            if (Eithers.isRight(runInstancesResult))
            {
                return AwsInstanceReservationDescription.create(runInstancesResult.getRight().getReservation());
            }

            subnetId = subnetChooser.choose(instanceConfiguration.getSubnets(), instanceType);
        } while (subnetId!=null && !subnetId.equals(firstSubnetId));
        throw runInstancesResult.getLeft();
    }

    @NotNull
    private EC2InstanceType getInstanceType(final InstanceLaunchConfiguration instanceConfiguration)
    {
        final Iterable<EC2InstanceType> instanceTypes = instanceConfiguration.getInstanceTypes();
        final EC2InstanceType instanceType = Iterables.get(instanceTypes, 0);
        if (Iterables.size(instanceTypes)>1)
        {
            log.info("Launching a non-spot fleet instance and the image configuration has more than one instance type defined, using " + instanceType);
        }
        return instanceType;
    }


    private Pair<AmazonServiceException,RunInstancesResult> runInstance(@Nullable final SubnetId subnet, final EC2InstanceType instanceType)
    {
        final RunInstancesRequest runInstancesRequest = getRunInstancesRequest(subnet, instanceType);

        log.info("Ordering EC2 instance of image " + instanceConfiguration.getImage().getId() + (subnet!=null? " in subnet " + subnet : ""));
        RunInstancesResult runInstancesResult;
        try
        {
            runInstancesResult = awsAccount.getAmazonEc2().runInstances(runInstancesRequest);
        }
        catch (final AmazonServiceException e)
        {
            if (!AmazonServiceErrorCode.INSUFFICIENT_INSTANCE_CAPACITY.is(e) || subnet==null)
            {
                throw e;
            }
            subnetChooser.blacklist(subnet, instanceType);
            return Eithers.left(e);
        }
        return Eithers.right(runInstancesResult);
    }

    private RunInstancesRequest getRunInstancesRequest(@Nullable final SubnetId subnetId, final EC2InstanceType instanceType)
    {
        final String imageId = instanceConfiguration.getImage().getId();

        final RunInstancesRequest runInstancesRequest =
                new RunInstancesRequest()
                        .withImageId(imageId)
                        .withKeyName(instanceConfiguration.getKeyName())
                        .withInstanceType(instanceType.getAwsInstanceType())
                        .withInstanceInitiatedShutdownBehavior(AwsSupportConstants.InstanceInitiatedShutdownBehaviour.TERMINATE.toString())
                        .withIamInstanceProfile(instanceConfiguration.getIamInstanceProfile())
                        .withEbsOptimized(instanceConfiguration.isEbsOptimised())
                        .withBlockDeviceMappings(getBlockDeviceMappings(awsAccount, imageId, instanceType, instanceConfiguration.getEbsSnapshotId()))
                        .withMinCount(1)
                        .withMaxCount(1)
                        .withClientToken(RandomStringUtils.randomAlphanumeric(64));

        final Iterable<SecurityGroupId> securityGroupIds = instanceConfiguration.getSecurityGroups(subnetId);
        if (isVpcEnabled(instanceConfiguration))
        {
            assert subnetId!=null;
            if (instanceConfiguration.shouldAssociatePublicIp())
            {
                runInstancesRequest.withNetworkInterfaces(newPublicIpForVpc(subnetId, securityGroupIds));
            }
            else
            {
                runInstancesRequest
                        .withSubnetId(subnetId.getId())
                        .withSecurityGroupIds(ResourceId.getIds(securityGroupIds));
            }
        }
        else
        {
            runInstancesRequest
                    .withPlacement(new Placement().withAvailabilityZone(instanceConfiguration.getRequestedAvailabilityZone()))
                    .withSecurityGroupIds(ResourceId.getIds(securityGroupIds));
        }

        runInstancesRequest.setUserData(EC2Utils.getUserDataAsString(instanceConfiguration.getUserData()));
        return runInstancesRequest;
    }
}