package com.atlassian.aws.ec2.awssdk;

import com.amazonaws.regions.Region;
import com.amazonaws.regions.RegionUtils;
import com.amazonaws.regions.Regions;
import org.jetbrains.annotations.NotNull;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

public class RegionSupport
{
    private static final Map<Region, String> REGION_NAMES = new HashMap<>();
    static
    {
        addRegion(Regions.CA_CENTRAL_1, "ca-c1");
        addRegion(Regions.US_EAST_1, "us-e1");
        addRegion(Regions.US_EAST_2, "us-e2");
        addRegion(Regions.US_WEST_1, "us-w1");
        addRegion(Regions.US_WEST_2, "us-w2");
        addRegion(Regions.EU_CENTRAL_1, "eu-c1");
        addRegion(Regions.EU_WEST_1, "eu-w1");
        addRegion(Regions.EU_WEST_2, "eu-w2");
        addRegion(Regions.EU_WEST_3, "eu-w3");
        addRegion(Regions.AP_SOUTHEAST_1, "ap-se1");
        addRegion(Regions.AP_SOUTHEAST_2, "ap-se2");
        addRegion(Regions.AP_NORTHEAST_1, "ap-ne1");
        addRegion(Regions.AP_NORTHEAST_2, "ap-ne2");
        addRegion(Regions.AP_SOUTH_1, "ap-s1");
        addRegion(Regions.SA_EAST_1, "sa-east-1");
        addRegion(Regions.GovCloud, "us-gov-w1");
        addRegion(Regions.CN_NORTH_1, "cn-n1");
        addRegion(Regions.CN_NORTHWEST_1, "cn-nw1");
    }

    private RegionSupport()
    {
    }

    private static void addRegion(final Regions region, final String name)
    {
        final String previous = REGION_NAMES.put(Region.getRegion(region), name);
        if (previous!=null)
        {
            throw new IllegalArgumentException("Duplicate value for " + region);
        }
    }

    public static String getNormalisedName(final Region region)
    {
        final String name = REGION_NAMES.get(region);
        if (name==null)
        {
            throw new IllegalArgumentException("Unknown region " + region);
        }
        return name;
    }

    public static String getNormalisedName(final Regions region)
    {
        final String name = REGION_NAMES.get(toRegion(region));
        if (name==null)
        {
            throw new IllegalArgumentException("Unknown region " + region);
        }
        return name;
    }

    public static Region toRegion(Regions region)
    {
        return RegionUtils.getRegion(region.getName());
    }

    @NotNull
    public static AwsSupportConstants.Region toAtlassianAwsRegion(final @NotNull Regions regionEnum)
    {
        final Region region = RegionSupport.toRegion(regionEnum);
        for (final AwsSupportConstants.Region atlRegion : AwsSupportConstants.Region.values())
        {
            if (Objects.equals(atlRegion.getSdkRegion(), region))
            {
                return atlRegion;
            }
        }
        throw new IllegalArgumentException("Unable to map " + regionEnum + " " + " to atlassian-aws region");
    }

    @NotNull
    public static Regions toAwsSdkRegion(final @NotNull AwsSupportConstants.Region atlRegion)
    {
        return Regions.fromName(atlRegion.getSdkRegion().getName());
    }
}
