package com.atlassian.aws.ec2.awssdk;

import com.amazonaws.services.ec2.model.InstanceType;
import com.amazonaws.services.ec2.model.SpotInstanceRequest;
import com.amazonaws.services.ec2.model.SpotInstanceStateFault;
import com.atlassian.aws.ec2.InstanceReservationDescription;
import com.atlassian.aws.ec2.model.SubnetId;
import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import org.apache.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;

/**
 * This class adapts AWS SDK's reservation description to InstanceReservationDescription interface.
 */
public class AwsSpotInstanceReservationDescription implements InstanceReservationDescription
{
    private static final Logger log = Logger.getLogger(AwsInstanceReservationDescription.class);
    private final SpotInstanceRequest spotInstanceRequest;

    public AwsSpotInstanceReservationDescription(final SpotInstanceRequest spotInstanceRequest)
    {
        this.spotInstanceRequest = spotInstanceRequest;
    }

    @NotNull
    public static Collection<InstanceReservationDescription> create(@NotNull final Collection<SpotInstanceRequest> spotInstanceRequests)
    {
        return Collections2.transform(spotInstanceRequests, new Function<SpotInstanceRequest, InstanceReservationDescription>()
        {
            @Override
            public InstanceReservationDescription apply(final SpotInstanceRequest spotInstanceRequest)
            {
                return new AwsSpotInstanceReservationDescription(spotInstanceRequest);
            }
        });
    }

    @Override
    public String getInstanceId()
    {
        return spotInstanceRequest.getInstanceId();
    }

    public String getSpotInstanceRequestId()
    {
        return spotInstanceRequest.getSpotInstanceRequestId();
    }

    @Override
    public String getAvailabilityZone()
    {
        return spotInstanceRequest.getLaunchedAvailabilityZone();
    }

    @Nullable
    @Override
    public SubnetId getSubnet()
    {
        return SubnetId.fromNullable(spotInstanceRequest.getLaunchSpecification().getSubnetId());
    }

    @NotNull
    @Override
    public String getAddress()
    {
        return "none";
    }

    @Nullable
    @Override
    public InstanceType getInstanceType()
    {
        return InstanceType.fromValue(spotInstanceRequest.getLaunchSpecification().getInstanceType());
    }

    @NotNull
    @Override
    public String getHostname()
    {
        return "none";
    }

    @Override
    public AwsSupportConstants.InstanceStateName getState()
    {
        return AwsSupportConstants.InstanceStateName.fromValue(spotInstanceRequest.getState());
    }

    @Override
    public String getStateDescription()
    {
        final SpotInstanceStateFault fault = spotInstanceRequest.getFault();
        
        final String description = spotInstanceRequest.getState();
        if (fault!=null)
        {
            return description + ", fault: " + fault;
        }
        else
        {
            return description;
        }

    }
}
