package com.atlassian.aws.ec2.awssdk;

import com.amazonaws.services.ec2.model.InstanceType;
import com.amazonaws.services.ec2.model.RequestSpotFleetResult;
import com.amazonaws.services.ec2.model.SpotFleetLaunchSpecification;
import com.amazonaws.services.ec2.model.SpotPlacement;
import com.atlassian.aws.ec2.InstanceReservationDescription;
import com.atlassian.aws.ec2.awssdk.AwsSupportConstants.InstanceStateName;
import com.atlassian.aws.ec2.model.SubnetId;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.collect.Iterables;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * This class adapts AWS SDK's reservation description to InstanceReservationDescription interface.
 */
public class AwsSpotFleetInstanceRequest implements InstanceReservationDescription
{
    private final Collection<SpotFleetLaunchSpecification> spotFleetLaunchSpecifications;
    private final String spotFleetRequestId;
    private final Supplier<String> availabilityZone = uniqSupplier(e -> {
        final SpotPlacement placement = e.getPlacement();
        return placement != null ? placement.getAvailabilityZone() : "n/a";
    });

    private final Supplier<SubnetId> subnetId = uniqSupplier(e -> SubnetId.fromNullable(e.getSubnetId()));

    private final Supplier<InstanceType> instanceType = uniqSupplier(launchSpec -> {
        final String instanceType1 = launchSpec.getInstanceType();
        return instanceType1 !=null ? InstanceType.fromValue(instanceType1) : null;
    });

    public AwsSpotFleetInstanceRequest(final String spotFleetRequestId, final Collection<SpotFleetLaunchSpecification> spotFleetLaunchSpecifications)
    {
        this.spotFleetRequestId = spotFleetRequestId;
        this.spotFleetLaunchSpecifications = spotFleetLaunchSpecifications;
    }

    @NotNull
    public static Collection<InstanceReservationDescription> create(final RequestSpotFleetResult requestSpotFleetResult, final Collection<SpotFleetLaunchSpecification> launchSpecifications)
    {
        return Collections.singleton(new AwsSpotFleetInstanceRequest(requestSpotFleetResult.getSpotFleetRequestId(), launchSpecifications));
    }

    @Override
    public String getInstanceId()
    {
        return null;
    }

    public String getSpotFleetRequestId()
    {
        return spotFleetRequestId;
    }

    @Override
    public String getAvailabilityZone()
    {
        return availabilityZone.get();
    }

    @Nullable
    @Override
    public SubnetId getSubnet()
    {
        return subnetId.get();
    }

    @NotNull
    @Override
    public String getAddress()
    {
        return "none";
    }

    @Nullable
    @Override
    public InstanceType getInstanceType()
    {
        return instanceType.get();
    }

    @NotNull
    @Override
    public String getHostname()
    {
        return "none";
    }

    @Override
    public AwsSupportConstants.InstanceStateName getState()
    {
        return InstanceStateName.SpotOpen;
    }

    @Override
    public String getStateDescription()
    {
        return getState().toString();
    }


    @NotNull
    private <T> Supplier<T> uniqSupplier(final Function<SpotFleetLaunchSpecification, T> converter)
    {
        return Suppliers.memoize(() -> uniq(converter));
    }

    @Nullable
    private <T> T uniq(final Function<SpotFleetLaunchSpecification, T> converter)
    {
        final Set<T> values = spotFleetLaunchSpecifications.stream()
                .map(converter)
                .collect(Collectors.toSet());
        if (values.size()==1)
        {
            return Iterables.getOnlyElement(values);
        }
        return null;
    }
}
