package com.atlassian.aws.ec2.awssdk;

import com.amazonaws.services.ec2.model.Instance;
import com.amazonaws.services.ec2.model.InstanceState;
import com.amazonaws.services.ec2.model.InstanceType;
import com.amazonaws.services.ec2.model.Reservation;
import com.atlassian.aws.ec2.InstanceReservationDescription;
import com.atlassian.aws.ec2.model.SubnetId;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.stream.Collectors;

/**
 * This class adapts AWS SDK's reservation description to InstanceReservationDescription interface.
 */
public class AwsInstanceReservationDescription implements InstanceReservationDescription
{
    private final Instance instance;

    public AwsInstanceReservationDescription(final Instance instance)
    {
        this.instance = instance;
    }

    @NotNull
    public static Collection<InstanceReservationDescription> create(final Reservation reservation)
    {
        return reservation.getInstances().stream()
                .map(AwsInstanceReservationDescription::new)
                .collect(Collectors.toList());
    }

    @Override
    public String getInstanceId()
    {
        return instance.getInstanceId();
    }

    @Override
    public String getAvailabilityZone()
    {
        return instance.getPlacement().getAvailabilityZone();
    }

    @Nullable
    @Override
    public SubnetId getSubnet()
    {
        return SubnetId.fromNullable(instance.getSubnetId());
    }

    @NotNull
    @Override
    public String getAddress()
    {
        final String publicIpAddress = instance.getPublicIpAddress(); //VPCs have no public DNS name, but can have EIP
        if (StringUtils.isNotBlank(publicIpAddress))
        {
            return publicIpAddress;
        }

        return instance.getPrivateIpAddress();
    }

    @Nullable
    @Override
    public InstanceType getInstanceType()
    {
        return InstanceType.fromValue(instance.getInstanceType());
    }

    @NotNull
    @Override
    public String getHostname()
    {
        final String publicDnsName = instance.getPublicDnsName();
        if (StringUtils.isNotBlank(publicDnsName))
        {
            return publicDnsName;
        }
        final String publicIpAddress = instance.getPublicIpAddress(); //VPCs have no public DNS name, but can have EIP
        if (StringUtils.isNotBlank(publicIpAddress))
        {
            return publicIpAddress;
        }
        final String privateDnsName = instance.getPrivateDnsName();
        if (StringUtils.isNotBlank(privateDnsName))
        {
            return privateDnsName;
        }
        return instance.getPrivateIpAddress(); //VPCs have no private DNS name, but can have a private IP
    }

    @Override
    public AwsSupportConstants.InstanceStateName getState()
    {
        return AwsSupportConstants.InstanceStateName.fromValue(instance.getState().getName());
    }

    @Override
    public String getStateDescription()
    {
        InstanceState state = instance.getState();
        return state.getName() + " (" + state.getCode() + ") for reason [" + instance.getStateTransitionReason() + "]";
    }

}
