package com.atlassian.aws.ec2;

import com.amazonaws.services.ec2.model.InstanceType;
import com.amazonaws.services.ec2.model.SpotPrice;
import org.apache.log4j.Logger;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Date;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

public class SpotPriceMatrix
{
    private static final Logger log = Logger.getLogger(SpotPriceMatrix.class);
    
    private final Map<String, EnumMap<InstanceType, Price>> spotPriceMap = new TreeMap<String, EnumMap<InstanceType, Price>>();

    public static class Price
    {
        private final Date timestamp;
        private final double spotPrice;

        public Price(final double spotPrice)
        {
            this(spotPrice, new Date());
        }

        public Price(final double spotPrice, final Date timestamp)
        {
            this.timestamp = timestamp;
            this.spotPrice = spotPrice;
        }

        public Date getTimestamp()
        {
            return timestamp;
        }

        public double getSpotPrice()
        {
            return spotPrice;
        }

        @Override
        public String toString()
        {
            return String.valueOf(spotPrice);
        }

    }

    public SpotPriceMatrix()
    {
    }

    public SpotPriceMatrix(final List<SpotPrice> spotPrices)
    {
        add(spotPrices);
    }


    public void add(final SpotPrice spotPrice)
    {
        String product = spotPrice.getProductDescription();

        try
        {
            InstanceType instanceType = InstanceType.fromValue(spotPrice.getInstanceType());
            Double price = Double.valueOf(spotPrice.getSpotPrice());

            add(product, instanceType, new Price(price, spotPrice.getTimestamp()));
        }
        catch (IllegalArgumentException ignored)
        {
            log.debug("Skipping spot price for " + spotPrice.getInstanceType());
        }
    }

    public void add(String product, InstanceType instanceType, double bid)
    {
        add(product, instanceType, new Price(bid));
    }

    public void add(String product, InstanceType instanceType, Price spotPrice)
    {
        EnumMap<InstanceType, Price> instancesPricesForProduct = spotPriceMap.get(product);
        if (instancesPricesForProduct == null)
        {
            instancesPricesForProduct = new EnumMap<InstanceType, Price>(InstanceType.class);
            spotPriceMap.put(product, instancesPricesForProduct);
        }
        
        instancesPricesForProduct.put(instanceType, spotPrice);
    }

    public void add(final Iterable<SpotPrice> spotPrices)
    {
        for (SpotPrice spotPrice : spotPrices)
        {
            add(spotPrice);
        }
    }

    @Nullable
    public Price getPrice(String product, InstanceType instanceType)
    {
        Map<InstanceType, Price> productPrice = spotPriceMap.get(product);
        if (productPrice == null)
        {
            return null;
        }

        return productPrice.get(instanceType);
    }

    public Collection<String> getProducts()
    {
        return spotPriceMap.keySet();
    }

}