package com.atlassian.aws.ec2;

import com.amazonaws.services.ec2.model.ActiveInstance;
import com.amazonaws.services.ec2.model.CreateTagsRequest;
import com.amazonaws.services.ec2.model.DescribeSpotFleetInstancesRequest;
import com.amazonaws.services.ec2.model.DescribeSpotFleetInstancesResult;
import com.amazonaws.services.ec2.model.Instance;
import com.amazonaws.services.ec2.model.InstanceType;
import com.amazonaws.services.ec2.model.SpotInstanceRequest;
import com.amazonaws.services.ec2.model.Tag;
import com.atlassian.aws.AWSAccount;
import com.atlassian.aws.AWSException;
import com.atlassian.aws.ec2.awssdk.AwsInstanceReservationDescription;
import com.atlassian.aws.ec2.awssdk.AwsSpotFleetActiveInstance;
import com.atlassian.aws.ec2.awssdk.AwsSpotFleetInstanceRequest;
import com.atlassian.aws.ec2.awssdk.AwsSpotInstanceReservationDescription;
import com.atlassian.aws.ec2.awssdk.AwsSupportConstants;
import com.atlassian.aws.ec2.awssdk.AbstractDelegatingInstanceReservationDescription;
import com.atlassian.aws.ec2.awssdk.AwsSupportConstants.InstanceStateName;
import com.atlassian.aws.ec2.awssdk.InstanceLauncher;
import com.atlassian.aws.ec2.awssdk.launch.InstanceLauncherFactory;
import com.atlassian.aws.ec2.caches.Ec2CacheMissException;
import com.google.common.collect.Iterables;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.Priority;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

public class RemoteEC2InstanceImpl implements RemoteEC2Instance
{
    private static final Logger log = Logger.getLogger(RemoteEC2InstanceImpl.class);

    static final int DEFAULT_TIMEOUT = 300;
    private static final int SHUTDOWN_TIMEOUT_IN_SECONDS = DEFAULT_TIMEOUT;

    private final AtomicInteger successiveSupervisionFailures = new AtomicInteger();
    private final int pollPeriodInSeconds;
    private final int maxSuccessiveSupervisionFailures;
    private final AWSAccount awsAccount;
    private final EC2InstanceListener listener;
    private final ScheduledExecutorService scheduledExecutorService;

    private final InstanceLaunchConfiguration instanceConfiguration;
    private final InstanceStatus instanceStatus;
    private static final long AWS_RESOURCE_ID_PROPAGATION_TIME = TimeUnit.SECONDS.toMillis(40); //time after which a resource id should be propagated across AWS cluster
    private final InstanceLauncherFactory instanceLauncherFactory;
    private volatile InstanceReservationDescription lastKnownReservationDescription;

    /**
     * <p>A decorator for {@link Runnable}s that logs and stores unhandled {@link Throwable}s.</p>
     */
    static class CatchingRunnableDecorator implements Runnable
    {
        private final String description;
        private final Runnable runnable;

        public CatchingRunnableDecorator(final String description, final Runnable runnable)
        {
            this.description = description;
            this.runnable = runnable;
        }

        @Override
        public void run()
        {
            try
            {
                log.trace("Entering " + description);
                runnable.run();
            }
            catch (final Throwable throwable)
            {
                log.error("Exception during " + description, throwable);
            }
        }
    }

    private final AtomicBoolean startCalled = new AtomicBoolean();

    private final Runnable launcherTask = new CatchingRunnableDecorator("backgroundStart()", new Runnable()
    {
        @Override
        public void run()
        {
            backgroundStart();
        }
    });

    private final Runnable supervisor = new CatchingRunnableDecorator("backgroundSupervise()", new Runnable()
    {
        @Override
        public void run()
        {
            backgroundSupervise();
        }
    });


    private final AtomicBoolean isBeingTerminated = new AtomicBoolean();

    private volatile ScheduledFuture<?> supervisorJob;
    private volatile EC2InstanceState state = EC2InstanceState.INITIAL;

    /**
     * @param pollPeriodInSeconds      The period of time (in seconds) to wait between requests for EC2 instance status
     *                                 updates.
     * @param maxSuccessiveSupervisionFailures
     *                                 The number of successive attempts to obtain EC2 instance status updates that are
     *                                 allowed to fail before the instance is considered to have failed, and an attempt
     *                                 is made to terminate it.
     * @param listener                 A listener to receive notifications of changes in the state of the instance.
     * @param scheduledExecutorService The executor service to be used to execute background tasks.
     */
    public RemoteEC2InstanceImpl(@NotNull final  InstanceLaunchConfiguration instanceConfiguration, final int pollPeriodInSeconds, final int maxSuccessiveSupervisionFailures,
                                 final EC2InstanceListener listener,
                                 final AWSAccount awsAccount,
                                 final ScheduledExecutorService scheduledExecutorService)
    {
        this.instanceConfiguration = instanceConfiguration;
        this.pollPeriodInSeconds = pollPeriodInSeconds;
        this.maxSuccessiveSupervisionFailures = maxSuccessiveSupervisionFailures;
        this.listener = listener;
        this.awsAccount = awsAccount;
        this.scheduledExecutorService = scheduledExecutorService;
        instanceStatus = new InstanceStatus();
        instanceLauncherFactory = new InstanceLauncherFactory(awsAccount);
    }

    @Override
    public void start()
    {
        if (!startCalled.compareAndSet(false, true))
        {
            throw new IllegalStateException("Already started.");
        }
        scheduledExecutorService.execute(launcherTask);
    }

    @Override
    public void asyncTerminate()
    {
        if (!startCalled.get())
        {
            throw new IllegalStateException("Not started.");
        }

        if (isBeingTerminated.compareAndSet(false, true))
        {
            InstanceTerminator.terminate(awsAccount, scheduledExecutorService, instanceStatus);
        }
    }

    @Override
    public boolean isBeingTerminated()
    {
        return isBeingTerminated.get();
    }

    InstanceReservationDescription describeInstance()
    {
        final boolean noInstanceId = instanceStatus.getInstanceId() == null;
        final String spotOrFleetInstanceRequestId = instanceStatus.getSpotInstanceRequestId();
        if (noInstanceId && spotOrFleetInstanceRequestId == null)
        {
            throw new IllegalStateException("The instance has neither an instance id nor a spot request id");
        }

        final InstanceReservationDescription newInstanceReservationDescription = noInstanceId ?
                describeSpotOrFleet(spotOrFleetInstanceRequestId) : describeInstance(lastKnownReservationDescription);

        lastKnownReservationDescription = newInstanceReservationDescription;
        return newInstanceReservationDescription;
    }

    private InstanceReservationDescription describeInstance(@Nullable final InstanceReservationDescription lastKnownDescription) {
        try {
            final Collection<Instance> describeInstancesResult = awsAccount.describeInstances(getInstanceId());
            final Instance instance = Iterables.getOnlyElement(describeInstancesResult);
            instanceStatus.setLaunchTime(instance.getLaunchTime());
            return new AwsInstanceReservationDescription(instance);
        } catch (final Ec2CacheMissException e) {
            if (lastKnownDescription==null || lastKnownDescription.getState() != InstanceStateName.ShuttingDown) {
                throw e;
            }
            return new AbstractDelegatingInstanceReservationDescription(lastKnownDescription) {
                @Override
                public InstanceStateName getState() {
                    return InstanceStateName.Terminated;
                }
            };
        }
    }

    @NotNull
    private InstanceReservationDescription describeSpotOrFleet(final String spotOrFleetInstanceRequestId)
    {
        if (spotOrFleetInstanceRequestId.startsWith("sfr-"))
        {
            final DescribeSpotFleetInstancesRequest describeSpotFleetInstancesRequest =
                    new DescribeSpotFleetInstancesRequest()
                    .withSpotFleetRequestId(spotOrFleetInstanceRequestId);
            final DescribeSpotFleetInstancesResult describeSpotFleetInstancesResult =
                    awsAccount.getAmazonEc2()
                            .describeSpotFleetInstances(describeSpotFleetInstancesRequest);
            final List<ActiveInstance> activeInstances = describeSpotFleetInstancesResult.getActiveInstances();
            if (!activeInstances.isEmpty())
            {
                final ActiveInstance activeInstance = Iterables.getOnlyElement(activeInstances);
                return new AwsSpotFleetActiveInstance(activeInstance);
            }
            else
            {
                return new AwsSpotFleetInstanceRequest(spotOrFleetInstanceRequestId, null);
            }
        }
        else
        {
            final SpotInstanceRequest describedSpotRequest = describeSpotRequest();

            if (StringUtils.isNotBlank(describedSpotRequest.getInstanceId()))
            {
                //according to the documentation, at this point the state should be Active or Closed, but it does not happen
                //we don't really care about the spot request state after the instance is launched, it's Active as far as we are
                //concerned
                describedSpotRequest.setState(AwsSupportConstants.InstanceStateName.SpotActive.toString());
                log.info("Spot instance request " + spotOrFleetInstanceRequestId + " is now active as instance " + describedSpotRequest.getInstanceId());
            }
            return new AwsSpotInstanceReservationDescription(describedSpotRequest);
        }
    }

    private SpotInstanceRequest describeSpotRequest()
    {
        Collection<SpotInstanceRequest> describeSpotInstanceRequestsResult = awsAccount.describeSpotInstanceRequests(instanceStatus.getSpotInstanceRequestId());

        return Iterables.getOnlyElement(describeSpotInstanceRequestsResult);
    }

    AWSException unexpectedStateException(final InstanceReservationDescription instance)
    {
        //noinspection ThrowableInstanceNeverThrown
        return new AWSException("EC2 instance " + instanceStatus.getSensibleId() + " in an unexpected state " + instance.getStateDescription());
    }

    public void handleAddressChange(@NotNull final InstanceReservationDescription instance)
    {
        final String previousAddress = instanceStatus.getAddress();
        final String newAddress = instance.getAddress();
        log.info("Bamboo has detected that the instance " + instance.getInstanceId() + ", previously available at " + previousAddress + " is now available at " + newAddress);
        instanceStatus.setAddressAndHostname(instance);
        listener.onInstanceAddressChange(previousAddress, newAddress);
    }

    public void handleStateChange(final InstanceReservationDescription instance, final AwsSupportConstants.InstanceStateName newState)
    {
        instanceStatus.setState(instance.getState());
        final long timeElapsed = instanceStatus.getSecondsSinceStartupAttempt();
        
        String msg = null;
        switch (newState)
        {
            case SpotActive:
                log.info("Spot request fulfilled after " + timeElapsed + " seconds.");
                updatePostLaunchData(instance);
                setSupervisionState(EC2InstanceState.PENDING, msg, null);
                break;
            case SpotCancelled:
            case SpotClosed:
            case SpotFailed:
            {
                final String statusDescription = "Spot request " + ((AwsSpotInstanceReservationDescription) instance).getSpotInstanceRequestId() + " state is " + newState + " after " + timeElapsed + " seconds,";
                if (instanceConfiguration.isSpotRequestTimeoutExpired(timeElapsed))
                {
                    log.info(statusDescription + ", falling back to a regular instance.");
                    launchInstance(instanceLauncherFactory.newOnDemandInstanceLauncher(instanceConfiguration, instanceStatus));
                }
                else
                {
                    log.info(statusDescription +", assuming this was a manual cancellation.");
                    setSupervisionState(EC2InstanceState.TERMINATED, msg, null);
                }
            }
                break;
            case Running:
                updatePostLaunchData(instance);
                final String address = instance.getAddress();
                instanceStatus.setAddressAndHostname(instance);
                msg = "Bamboo has detected that EC2 instance " + getInstanceId() + " is now running at " + address;
                setSupervisionState(EC2InstanceState.RUNNING, msg, null);
                break;
            case Stopping:
                msg = "Bamboo has detected that EC2 EBS-backed instance " + getInstanceId() + " is stopping.";
                setSupervisionState(EC2InstanceState.STOPPING, msg, null);
                break;
            case Stopped:
                msg = "Bamboo has detected that EC2 EBS-backed instance " + getInstanceId() + " has stopped.";
                setSupervisionState(EC2InstanceState.STOPPED, msg, null);
                break;
            case ShuttingDown:
                msg = "Bamboo has detected that EC2 instance " + getInstanceId() + " is shutting down. State: " + instance.getStateDescription();
                instanceStatus.setDeadline(SHUTDOWN_TIMEOUT_IN_SECONDS);
                setSupervisionState(EC2InstanceState.SHUTTING_DOWN, msg, null);
                break;
            case Terminated:
                msg = "EC2 instance " + getInstanceId() + " has terminated.";
                setSupervisionState(EC2InstanceState.TERMINATED, msg, null);
                break;
        }

        if (msg!=null)
        {
            log.info(msg);
        }
    }

    private void updatePostLaunchData(final InstanceReservationDescription instance)
    {
        instanceStatus.setInstanceId(instance.getInstanceId());
        instanceStatus.setAvailabilityZone(instance.getAvailabilityZone());
        instanceStatus.setState(instance.getState());
        final InstanceType instanceType = instance.getInstanceType();
        if (instanceType!=null)
        {
            instanceStatus.setInstanceType(instanceType);
        }
    }

    private synchronized void backgroundStart()
    {
        log.trace("Entered backgroundStart()");

        InstanceLauncher launcher = instanceLauncherFactory.newLauncher(instanceConfiguration, instanceStatus);

        try
        {
            if (launchInstance(launcher))
            {
                supervisorJob = scheduledExecutorService.scheduleWithFixedDelay(supervisor, 0, pollPeriodInSeconds, TimeUnit.SECONDS);
            }
        }
        finally
        {
            log.trace("Finished backgroundStart()");
        }
    }

    private boolean launchInstance(final InstanceLauncher launcher)
    {
        instanceStatus.onStartupAttempt();
        final InstanceReservationDescription instance;
        try
        {
            final Collection<InstanceReservationDescription> instances = launcher.call();
            instance = Iterables.getOnlyElement(instances);
            instanceStatus.setSubnetId(instance.getSubnet());
        }
        catch (final Throwable throwable)
        {
            final String details = "EC2 instance order for image " + instanceConfiguration.getImage().getId() + " failed.";
            log.error(details, throwable);
            setSupervisionState(EC2InstanceState.FAILED_TO_START, details, throwable);
            return false;
        }

        final String details;
        final EC2InstanceState initialState;
        if (instanceStatus.getInstancePaymentType() == InstancePaymentType.SPOT)
        {
            details = "Placed spot request " + instanceStatus.getSensibleId();
            initialState = EC2InstanceState.BIDDING;
        }
        else
        {
            instanceStatus.setInstanceId(instance.getInstanceId());
            details = "Ordered EC2 instance " + instanceStatus.getSensibleId();
            initialState = EC2InstanceState.PENDING;
        }

        instanceStatus.setInstanceType(instance.getInstanceType());
        instanceStatus.setAvailabilityZone(instance.getAvailabilityZone());
        log.info(details);

        setSupervisionState(initialState, details, null);
        return true;
    }

    private synchronized void backgroundSupervise()
    {
        log.trace("Entered backgroundSupervise()");
        try
        {
            if (state.isFinal())
            {
                throw new IllegalStateException(state + " is a final state.");
            }
            try
            {
                state.supervise(this);
                successiveSupervisionFailures.set(0);
            }
            catch (final Throwable throwable)
            {
                if (successiveSupervisionFailures.incrementAndGet() > maxSuccessiveSupervisionFailures)
                {
                    log.error("The request for the current status of EC2 instance/spot request " + instanceStatus.getSensibleId() + " failed after " + maxSuccessiveSupervisionFailures + " attempts.  No further attempts will be made.", throwable);
                    state.supervisionFailure(this, throwable);
                }
                else
                {
                    final Priority level = instanceStatus.isSensibleIdOlderThan(AWS_RESOURCE_ID_PROPAGATION_TIME) ? Level.WARN : Level.DEBUG;

                    log.log(level, "The request for the current status of EC2 instance/spot request " + instanceStatus.getSensibleId() + " failed, try " + successiveSupervisionFailures.get() + ".  Will retry later.", throwable);
                }
            }
        }
        finally
        {
            log.trace("Finished backgroundSupervise()");
        }
    }

    void setSupervisionState(final EC2InstanceState newState, @Nullable final String details, @Nullable final Throwable throwable)
    {
        if (newState == this.state) {
            return;
        }
        final EC2InstanceState previousState = this.state;
        this.state = newState;
        if (newState.isFinal())
        {
            if (supervisorJob != null)
            {
                log.debug("Cancelling supervisor");
                supervisorJob.cancel(false);
            }
        }
        final CatchingRunnableDecorator catchingRunnableDecorator = new CatchingRunnableDecorator("Listener " + listener,
                () -> listener.ec2InstanceStateChanged(RemoteEC2InstanceImpl.this, previousState, newState, details, throwable)
        );
        catchingRunnableDecorator.run();
    }

    @Override
    public InstanceLaunchConfiguration getInstanceConfiguration()
    {
        return instanceConfiguration;
    }

    @Override
    public InstanceStatus getInstanceStatus()
    {
        return instanceStatus;
    }

    @Override
    public String getInstanceId()
    {
        return instanceStatus.getInstanceId();
    }

    @Override
    public void addTag(@NotNull final String key, @NotNull final String value)
    {
        final CreateTagsRequest createTagsRequest =
                new CreateTagsRequest().
                        withResources(instanceStatus.getInstanceId()).
                        withTags(new Tag(key, value));

        awsAccount.getAmazonEc2().createTagsAsync(createTagsRequest);
    }

    @Override
    public void reconnectToRunningInstance(@NotNull final Instance instance)
    {
        state = EC2InstanceState.RUNNING;
        startCalled.compareAndSet(false, true);
        instanceStatus.setInstanceType(InstanceType.fromValue(instance.getInstanceType()));
        instanceStatus.setInstanceId(instance.getInstanceId());
        instanceStatus.setLaunchTime(instance.getLaunchTime());
        instanceStatus.setState(AwsSupportConstants.InstanceStateName.fromValue(instance.getState()));
        instanceStatus.setStartupTime(instance.getLaunchTime().getTime()); //not a bad estimation...well, the best we have

        final InstanceReservationDescription instanceReservationDescription = new AwsInstanceReservationDescription(instance);

        instanceStatus.setAvailabilityZone(instanceReservationDescription.getAvailabilityZone());
        instanceStatus.setAddressAndHostname(instanceReservationDescription);
        instanceStatus.setSubnetId(instanceReservationDescription.getSubnet());
        if (StringUtils.isNotBlank(instance.getSpotInstanceRequestId()))
        {
            instanceStatus.setSpotInstanceRequestId(instance.getSpotInstanceRequestId());
            instanceStatus.setInstancePaymentType(InstancePaymentType.SPOT);
        }
        else
        {
            instanceStatus.setInstancePaymentType(InstancePaymentType.REGULAR);
        }

        supervisorJob = scheduledExecutorService.scheduleWithFixedDelay(supervisor, 0, pollPeriodInSeconds, TimeUnit.SECONDS);
    }
}
