package com.atlassian.aws.ec2;

import com.amazonaws.AmazonServiceException;
import com.amazonaws.services.ec2.AmazonEC2;
import com.amazonaws.services.ec2.model.ActiveInstance;
import com.amazonaws.services.ec2.model.AmazonEC2Exception;
import com.amazonaws.services.ec2.model.CancelSpotFleetRequestsRequest;
import com.amazonaws.services.ec2.model.CancelSpotInstanceRequestsRequest;
import com.amazonaws.services.ec2.model.DescribeSpotFleetInstancesRequest;
import com.amazonaws.services.ec2.model.DescribeSpotFleetInstancesResult;
import com.amazonaws.services.ec2.model.DescribeSpotFleetRequestsRequest;
import com.amazonaws.services.ec2.model.DescribeSpotFleetRequestsResult;
import com.amazonaws.services.ec2.model.DescribeSpotInstanceRequestsRequest;
import com.amazonaws.services.ec2.model.DescribeSpotInstanceRequestsResult;
import com.amazonaws.services.ec2.model.InstanceStateChange;
import com.amazonaws.services.ec2.model.SpotFleetRequestConfig;
import com.amazonaws.services.ec2.model.SpotInstanceRequest;
import com.amazonaws.services.ec2.model.TerminateInstancesRequest;
import com.amazonaws.services.ec2.model.TerminateInstancesResult;
import com.atlassian.aws.AWSAccount;
import com.atlassian.aws.AmazonServiceErrorCode;
import com.atlassian.aws.ec2.RemoteEC2InstanceImpl.CatchingRunnableDecorator;
import com.atlassian.aws.ec2.awssdk.AwsSupportConstants;
import com.atlassian.aws.ec2.awssdk.AwsSupportConstants.SpotFleetRequestState;
import com.atlassian.aws.ec2.awssdk.AwsSupportConstants.SpotInstanceRequestState;
import com.atlassian.aws.ec2.model.SpotFleetRequestId;
import com.atlassian.aws.ec2.model.SpotRequestId;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.base.Throwables;
import com.google.common.collect.Iterables;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import static com.atlassian.aws.AmazonServiceErrorCode.UNAUTHORISED_OPERATION;

class InstanceTerminator extends CatchingRunnableDecorator
{
    private static final Logger log = Logger.getLogger(InstanceTerminator.class);

    InstanceTerminator(final AWSAccount awsAccount, final InstanceStatus instanceStatus, final Runnable taskCanceller)
    {
        super("terminator()", newTerminator(awsAccount, instanceStatus, taskCanceller));
    }

    private static boolean terminateInstance(final AmazonEC2 amazonEc2, @Nullable final String instanceId)
    {
        if (instanceId == null)
        {
            return true;
        }

        log.info("Requesting that EC2 instance " + instanceId + " be shut down.");
        final TerminateInstancesRequest terminateInstacesRequest = new TerminateInstancesRequest().withInstanceIds(instanceId);
        final TerminateInstancesResult terminateInstancesResult = amazonEc2.terminateInstances(terminateInstacesRequest);
        final List<InstanceStateChange> terminatingInstances = terminateInstancesResult.getTerminatingInstances();
        for (final InstanceStateChange terminatingInstance : terminatingInstances)
        {
            log.info("Requested transition of EC2 instance " + terminatingInstance.getInstanceId() + ", state will change from : " + terminatingInstance.getPreviousState() + " to " + terminatingInstance.getCurrentState() + ")");
        }

        return !terminatingInstances.isEmpty();
    }

    private static boolean cancelSpotFleetRequest(final AmazonEC2 amazonEc2, final InstanceStatus instanceStatus, final String spotFleetRequestId)
    {
        {
            final CancelSpotFleetRequestsRequest cancelRequest = new CancelSpotFleetRequestsRequest()
                    .withSpotFleetRequestIds(spotFleetRequestId);
            try
            {
                amazonEc2.cancelSpotFleetRequests(cancelRequest);
                log.info("Requested cancellation of spot request " + spotFleetRequestId);
            }
            catch (final AmazonServiceException e)
            {
                log.warn("Error when cancelling the request", e);
            }
        }

        {
            final DescribeSpotFleetInstancesRequest describeSpotFleetInstancesRequest =
                    new DescribeSpotFleetInstancesRequest().withSpotFleetRequestId(spotFleetRequestId);
            final DescribeSpotFleetInstancesResult describeSpotFleetInstancesResult = amazonEc2.describeSpotFleetInstances(describeSpotFleetInstancesRequest);
            final ActiveInstance activeInstance = Iterables.getOnlyElement(describeSpotFleetInstancesResult.getActiveInstances(), null);
            if (activeInstance!=null)
            {
                instanceStatus.setInstanceId(activeInstance.getInstanceId());
            }
        }

        {
            final DescribeSpotFleetRequestsRequest describeSpotInstanceRequestsRequest =
                    new DescribeSpotFleetRequestsRequest()
                            .withSpotFleetRequestIds(spotFleetRequestId);

            final DescribeSpotFleetRequestsResult describeSpotInstanceRequestsResult;
            try
            {
                describeSpotInstanceRequestsResult =
                        amazonEc2.describeSpotFleetRequests(describeSpotInstanceRequestsRequest);
            }
            catch (final AmazonServiceException e)
            {
                AmazonServiceErrorCode.INVALID_SPOT_FLEET_REQUEST_ID_NOT_FOUND.rethrowIfNot(e);
                return true;
            }
            final SpotFleetRequestConfig describedSpotFleetRequest = Iterables.getOnlyElement(describeSpotInstanceRequestsResult.getSpotFleetRequestConfigs());

            return SpotFleetRequestState.isFinal(describedSpotFleetRequest.getSpotFleetRequestState());
        }
    }

    private static boolean cancelSpotInstanceRequest(final AmazonEC2 amazonEc2, final InstanceStatus instanceStatus, final String spotInstanceRequestId)
    {
        {
            final CancelSpotInstanceRequestsRequest cancelRequest = new CancelSpotInstanceRequestsRequest()
                    .withSpotInstanceRequestIds(spotInstanceRequestId);
            try
            {
                amazonEc2.cancelSpotInstanceRequests(cancelRequest);
                log.info("Requested cancellation of spot request " + spotInstanceRequestId);
            }
            catch (final AmazonServiceException e)
            {
                log.warn("Error when cancelling the request", e);
            }
        }

        {
            final DescribeSpotInstanceRequestsRequest describeSpotInstanceRequestsRequest =
                    new DescribeSpotInstanceRequestsRequest()
                            .withSpotInstanceRequestIds(spotInstanceRequestId);

            final DescribeSpotInstanceRequestsResult describeSpotInstanceRequestsResult;
            try
            {
                describeSpotInstanceRequestsResult =
                        amazonEc2.describeSpotInstanceRequests(describeSpotInstanceRequestsRequest);
            }
            catch (final AmazonServiceException e)
            {
                AmazonServiceErrorCode.INVALID_SPOT_INSTANCE_REQUEST_ID_NOT_FOUND.rethrowIfNot(e);
                return true;
            }

            final SpotInstanceRequest describedSpotRequest = Iterables.getOnlyElement(describeSpotInstanceRequestsResult.getSpotInstanceRequests());
            final String describedInstanceId = describedSpotRequest.getInstanceId();
            if (StringUtils.isNotBlank(describedInstanceId))
            {
                instanceStatus.setInstanceId(describedInstanceId);
            }

            return SpotInstanceRequestState.isFinal(describedSpotRequest.getState());
        }
    }

    private static boolean cancelSpotRequest(final AmazonEC2 amazonEc2, final InstanceStatus instanceStatus)
    {
        //we don't have to cancel if there already is an instance or we already cancelled or we are not a spot instance
        final String spotInstanceRequestId = instanceStatus.getSpotInstanceRequestId();
        final String instanceId = instanceStatus.getInstanceId();
        if (instanceId!=null || spotInstanceRequestId==null)
        {
            return true;
        }
        if (SpotRequestId.isValid(spotInstanceRequestId))
        {
            return cancelSpotInstanceRequest(amazonEc2, instanceStatus, spotInstanceRequestId);
        }
        if (SpotFleetRequestId.isValid(spotInstanceRequestId))
        {
            return cancelSpotFleetRequest(amazonEc2, instanceStatus, spotInstanceRequestId);
        }
        throw new IllegalArgumentException("Unknown spot instance request id " + spotInstanceRequestId);
    }

    @NotNull
    private static Runnable newTerminator(final AWSAccount awsAccount, final InstanceStatus instanceStatus, final Runnable selfCanceller)
    {
        return new Runnable()
        {
            private volatile boolean isSpotRequestClosed = instanceStatus.getSpotInstanceRequestId()==null;

            @Override
            public void run()
            {
                runTermination();
                selfCanceller.run();
            }

            private void runTermination()
            {
                if (isTerminated(instanceStatus.getState()))
                {
                    log.info("Instance " + instanceStatus.getState() + " is already terminated.");
                    return;
                }
                
                log.debug("Started termination task for instance/spot request " + instanceStatus.getSensibleId());
                try
                {
                    if (!isSpotRequestClosed)
                    {
                        isSpotRequestClosed = cancelSpotRequest(awsAccount.getAmazonEc2(), instanceStatus);
                    }

                    final boolean isInstanceTerminated = terminateInstance(awsAccount.getAmazonEc2(), instanceStatus.getInstanceId());

                    if (isInstanceTerminated && isSpotRequestClosed)
                    {
                        //HAPPY!
                        log.debug("Instance " + instanceStatus.getSensibleId() + " has been terminated");
                    }
                }
                catch (final AmazonEC2Exception e)
                {
                    if (UNAUTHORISED_OPERATION.is(e))
                    {
                        //we can't do anything...
                        log.error("Bamboo Server does not have permissions to terminate " + instanceStatus.getSensibleId() + ". Leaving instance running, you are responsible for its termination.");
                    }
                    else
                    {
                        log.warn("Failed to order cancellation/termination of instance " + instanceStatus.getSensibleId() + ".  Will retry.", e);
                        throw e;   
                    }
                }
                catch (final RuntimeException e)
                {
                    log.warn("Failed to order cancellation/termination of instance " + instanceStatus.getSensibleId() + ".  Will retry.", e);
                    throw e;
                }
            }

            private boolean isTerminated(final AwsSupportConstants.InstanceStateName state)
            {
                return state == AwsSupportConstants.InstanceStateName.Terminated ||
                       state == AwsSupportConstants.InstanceStateName.ShuttingDown;
            }
        };
    }

    /**
     * Starts a background termination thread. Capable of cancelling spot/spot fleet requests and regular instances.
     */
    public static void terminate(final AWSAccount awsAccount, final ScheduledExecutorService scheduledExecutorService, final InstanceStatus instanceStatus)
    {
        final BlockingDeque<Future<?>> queue = new LinkedBlockingDeque<>(1);
        final Supplier<Future<?>> ref = Suppliers.memoize(new Supplier<Future<?>>()
        {
            @Override
            public Future<?> get()
            {
                try
                {
                    return queue.takeFirst();
                }
                catch (final InterruptedException e)
                {
                    throw Throwables.propagate(e);
                }
            }
        });

        final Runnable taskCanceller = () -> {
            final boolean mayInterruptIfRunning = false;
            ref.get().cancel(mayInterruptIfRunning);
        };

        final InstanceTerminator terminator = new InstanceTerminator(awsAccount, instanceStatus, taskCanceller);
        queue.add(scheduledExecutorService.scheduleWithFixedDelay(terminator, 0, 5, TimeUnit.MINUTES));
    }
}
