package com.atlassian.aws.ec2;

import com.amazonaws.services.ec2.model.InstanceType;
import com.atlassian.aws.ec2.awssdk.AwsSupportConstants;
import com.atlassian.aws.ec2.model.SubnetId;
import com.google.common.base.Preconditions;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Date;
import java.util.concurrent.TimeUnit;

public class InstanceStatus
{
    private volatile long deadline;
    private volatile InstancePaymentType instancePaymentType;
    private volatile String spotInstanceRequestId; //valid only for spot instances
    private volatile String instanceId;
    private volatile Date launchTime;
    /**
     * Availability zone as returned by EC2
     */
    private volatile String availabilityZone;

    /**
     * @return The fully-qualified public domain name of this EC2 instance, or <code>null</code> if it is not yet
     * available.
     */
    private String address;

    private String hostname;
    /**
     * Stopwatch used for timing spot requests and instance launches
     */
    public long startupTime = 0;
    private long sensibleIdCreationTime;
    private SubnetId subnetId;
    private InstanceType instanceType;
    private volatile AwsSupportConstants.InstanceStateName state;

    public void setDeadline(final int timeoutInSeconds)
    {
        deadline = System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(timeoutInSeconds);
    }

    boolean isDeadlinePassed()
    {
        return System.currentTimeMillis() > deadline;
    }

    public InstancePaymentType getInstancePaymentType()
    {
        return instancePaymentType;
    }

    public void setInstancePaymentType(InstancePaymentType instancePaymentType)
    {
        this.instancePaymentType = instancePaymentType;
    }

    public String getSpotInstanceRequestId()
    {
        return spotInstanceRequestId;
    }


    public void setSpotInstanceRequestId(String spotInstanceRequestId)
    {
        if (this.spotInstanceRequestId==null && instanceId==null)
        {
            sensibleIdCreationTime = System.currentTimeMillis();
        }
        this.spotInstanceRequestId = spotInstanceRequestId;
    }

    public String getInstanceId()
    {
        return instanceId;
    }

    public void setInstanceId(final String instanceId)
    {
        //noinspection VariableNotUsedInsideIf
        if (instanceId==null)
        {
            sensibleIdCreationTime = System.currentTimeMillis();
        }
        this.instanceId = instanceId;
    }

    /**
     * Returns either the instance id (if already available) or spot request id
     *
     * @return
     */
    @Nullable
    public String getSensibleId()
    {
        return instanceId != null ? instanceId : spotInstanceRequestId;
    }

    public String getAvailabilityZone()
    {
        return availabilityZone;
    }

    public void setAvailabilityZone(String availabilityZone)
    {
        this.availabilityZone = availabilityZone;
    }

    public String getAddress()
    {
        return address;
    }

    public void setAddressAndHostname(@NotNull final InstanceReservationDescription instance)
    {
        address = instance.getAddress();
        hostname = instance.getHostname();
    }

    public String getHostname()
    {
        return hostname;
    }

    /**
     * @return The time at which this EC2 instance was launched.
     */
    public Date getLaunchTime()
    {
        return launchTime;
    }

    public void setLaunchTime(Date launchTime)
    {
        this.launchTime = launchTime;
    }

    public void onStartupAttempt()
    {
        startupTime = System.currentTimeMillis();
    }

    public void setStartupTime(final long startupTime)
    {
        this.startupTime = startupTime;
    }

    public long getSecondsSinceStartupAttempt()
    {
        Preconditions.checkState(startupTime != 0);
        return TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis() - startupTime);
    }

    public boolean isSensibleIdOlderThan(final long awsResourceIdPropagationTimeMs)
    {
        return sensibleIdCreationTime+awsResourceIdPropagationTimeMs < System.currentTimeMillis();
    }

    public void setSubnetId(@Nullable final SubnetId subnetId)
    {
        this.subnetId = subnetId;
    }

    public InstanceType getInstanceType()
    {
        return instanceType;
    }

    public void setInstanceType(final InstanceType instanceType)
    {
        this.instanceType = instanceType;
    }

    public AwsSupportConstants.InstanceStateName getState()
    {
        return state;
    }

    public void setState(final AwsSupportConstants.InstanceStateName state)
    {
        this.state = state;
    }
}
