package com.atlassian.aws.ec2;

import com.amazonaws.services.ec2.model.IamInstanceProfileSpecification;
import com.atlassian.aws.ec2.configuration.EC2Image;
import com.atlassian.aws.ec2.configuration.EC2ImageImpl;
import com.atlassian.aws.ec2.model.SecurityGroupId;
import com.atlassian.aws.ec2.model.SubnetId;
import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.Iterables;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;

public class InstanceLaunchConfiguration
{
    private final EC2Image image;
    private final Iterable<String> requestedAvailabilityZones;
    private final int startupTimeoutInSeconds;
    private final double spotInstanceBid;
    private final Iterable<EC2InstanceType> instanceTypes;
    private final Iterable<SubnetId> subnets;
    private final int spotRequestTimeoutSeconds;
    private final String keyName;
    private final ImmutableMultimap<SubnetId, SecurityGroupId> availableSecurityGroups;
    private final Object userData;
    private final IamInstanceProfileSpecification iamInstanceProfile;
    private final String ebsSnapshotId;
    private final boolean shouldAssociatePublicIp;
    private final boolean ebsOptimised;

    public InstanceLaunchConfiguration(final String imageId,
                                       final String keyName,
                                       final ImmutableMultimap<SubnetId, SecurityGroupId> availableSecurityGroups,
                                       final Iterable<String> requestedAvailabilityZones,
                                       final int startupTimeoutInSeconds,
                                       final Collection<String> subnetIds,
                                       final int spotRequestTimeoutSeconds,
                                       final Object userData,
                                       final double spotInstanceBid,
                                       final Iterable<EC2InstanceType> instanceTypes,
                                       @Nullable final IamInstanceProfileSpecification iamInstanceProfile,
                                       @Nullable final String ebsSnapshotId, final boolean ebsOptmised,
                                       final boolean shouldAssociatePublicIp)
    {
        this.iamInstanceProfile = iamInstanceProfile;
        this.ebsSnapshotId = ebsSnapshotId;
        this.image = new EC2ImageImpl(imageId);
        this.requestedAvailabilityZones = requestedAvailabilityZones;
        this.startupTimeoutInSeconds = startupTimeoutInSeconds;
        this.subnets = subnetIds==null ? Collections.emptyList() : SubnetId.from(subnetIds);
        this.spotRequestTimeoutSeconds = spotRequestTimeoutSeconds;
        this.keyName = keyName;
        this.availableSecurityGroups = availableSecurityGroups;
        this.userData = userData;
        this.spotInstanceBid = spotInstanceBid;
        this.instanceTypes = instanceTypes == null ? Collections.singleton(EC2InstanceType.DEFAULT) : instanceTypes;
        this.ebsOptimised = ebsOptmised;
        this.shouldAssociatePublicIp = shouldAssociatePublicIp;
    }

    public double getSpotInstanceBid()
    {
        return spotInstanceBid;
    }

    public int getStartupTimeoutInSeconds()
    {
        return startupTimeoutInSeconds;
    }

    public EC2Image getImage()
    {
        return image;
    }

    public String getKeyName()
    {
        return keyName;
    }

    @NotNull
    public Iterable<SecurityGroupId> getSecurityGroups(@Nullable final SubnetId subnetId)
    {
        return availableSecurityGroups.get(MoreObjects.firstNonNull(subnetId, SubnetId.NO_SUBNET));
    }

    public Iterable<EC2InstanceType> getInstanceTypes()
    {
        return instanceTypes;
    }

    public String getRequestedAvailabilityZone()
    {
        return Iterables.getFirst(requestedAvailabilityZones, null);
    }

    @NotNull
    public Iterable<SubnetId> getSubnets()
    {
        return subnets;
    }

    public Object getUserData()
    {
        return userData;
    }

    public int getSpotRequestTimeoutSeconds()
    {
        return spotRequestTimeoutSeconds;
    }

    public boolean isSpotRequestTimeoutExpired(final long timeElapsed)
    {
        return timeElapsed>spotRequestTimeoutSeconds;
    }

    @Nullable
    public IamInstanceProfileSpecification getIamInstanceProfile()
    {
        return iamInstanceProfile;
    }

    @Nullable
    public String getEbsSnapshotId()
    {
        return ebsSnapshotId;
    }

    public boolean isEbsOptimised()
    {
        return ebsOptimised;
    }

    public boolean shouldAssociatePublicIp()
    {
        return shouldAssociatePublicIp;
    }
}
