package com.atlassian.aws.ec2;

import com.amazonaws.services.ec2.model.Image;
import com.amazonaws.services.ec2.model.IpPermission;
import com.amazonaws.services.ec2.model.SecurityGroup;
import com.amazonaws.util.EC2MetadataUtils;
import com.atlassian.aws.ec2.awssdk.AwsSupportConstants;
import com.atlassian.aws.ec2.configuration.EC2Image;
import com.atlassian.aws.ec2.configuration.EC2ImageImpl;
import com.atlassian.aws.utils.CompressionUtils;
import com.atlassian.aws.utils.JsonUtils;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.openssl.PEMKeyPair;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;
import java.util.StringTokenizer;

public class EC2Utils
{
    private static boolean isCompressionEnabled = false;

    private EC2Utils()
    {
    }

    /**
     * @return The {@link EC2Instance} upon which the current JVM is hosted.
     * @throws IOException if necessary data could not be read.
     */
    @NotNull
    public static EC2Instance getLocalEC2Instance() throws IOException
    {
        return Ec2UtilsThin.getLocalEC2Instance();
    }

    /**
     * <p>Retrieves an object from the EC2 user data for the current host.</p>
     *
     * @return The retrieved object.
     * @throws IOException            If the user data could not be read or deserialised.
     * @throws ClassNotFoundException If a required {@link Class} could not be found.
     */
    public static <T> T getUserData(Class<T> aClass) throws IOException, ClassNotFoundException
    {
        return Ec2UtilsThin.getUserData(aClass);
    }

    /**
     * @return The {@link EC2Image} upon which the current process is running.
     * @throws IOException If the {@link EC2Image} cannot be identified, possibly because the current process is not running in EC2.
     */
    public static EC2Image getCurrentImage() throws IOException
    {
        return new EC2ImageImpl(EC2MetadataUtils.getAmiId());
    }

    @Contract("!null -> !null")
    public static String getUserDataAsString(final Object userData)
    {
        if (userData==null)
        {
            return null;
        }

        return toUserDataString(userData);
    }

    private static String toUserDataString(final Object userData)
    {
        final String userDataJson = JsonUtils.toJson(userData);
        byte[] userDataJsonBytes = userDataJson.getBytes(StandardCharsets.UTF_8);

        if (!isCompressionEnabled)
        {
            userDataJsonBytes = CompressionUtils.compress(userDataJsonBytes);
        }
        return Base64.getMimeEncoder().encodeToString(userDataJsonBytes);
    }

    @Deprecated
    static void enableCompresssion()
    {
        isCompressionEnabled = true;
    }

    @Deprecated
    public static void disableCompresssion()
    {
        isCompressionEnabled = false;
    }

    public static boolean isWindows(@NotNull final Image image)
    {
        final String platform = image.getPlatform();
        return platform!=null && AwsSupportConstants.Platform.fromAwsName(platform).isWindows();
    }

    @NotNull
    public static List<IpPermission> getMatchingIpPermissions(@NotNull final SecurityGroup group, @NotNull final Protocol protocol, @Nullable final String cidrIpRange, final int port)
    {
        final List<IpPermission> ipPermissions = new ArrayList<>();
        for (final IpPermission permission : group.getIpPermissions())
        {
            final boolean protocolMatches = permission.getIpProtocol().equalsIgnoreCase(protocol.getValue());
            if (protocolMatches)
            {
                final int fromPort = MoreObjects.firstNonNull(permission.getFromPort(), -1);
                final int toPort = MoreObjects.firstNonNull(permission.getToPort(), -1);
                final boolean portMatches = fromPort<=port && port<=toPort;
                final boolean cidrMatches = cidrIpRange==null || permission.getIpRanges().contains(cidrIpRange);

                if (portMatches && cidrMatches)
                {
                    ipPermissions.add(permission);
                }
            }
        }
        return ipPermissions;
    }

    public static String getEbsDeviceName(@NotNull final Image image)
    {
        return EC2Utils.isWindows(image)? "xvdh" : "/dev/sdh";
    }

    /**
     * Parses a fingerprint in the xx:xx:xx:xx format into an array of bytes.
     */
    public static byte[] parseFingerprint(@NotNull final String fingerprintString)
    {
        Preconditions.checkArgument(StringUtils.isNotBlank(fingerprintString));
        final StringTokenizer stringTokenizer = new StringTokenizer(fingerprintString, ":");
        final byte[] bytes = new byte[stringTokenizer.countTokens()];
        for (int i = 0; i <bytes.length; i++)
        {
            bytes[i] = (byte)Integer.parseInt(stringTokenizer.nextToken(), 16);
        }
        return bytes;
    }

    public static boolean isKeyMatching(@NotNull final PEMKeyPair keypair, @NotNull final byte[] awsFingerprint) throws IOException
    {
        final byte[] local;
        if (awsFingerprint.length==16)
        {
            //MD5, public key
            final SubjectPublicKeyInfo publicKeyInfo = keypair.getPublicKeyInfo();
            local = DigestUtils.md5(publicKeyInfo.getEncoded());
        }
        else if (awsFingerprint.length==20)
        {
            //SHA-1, private key
            final PrivateKeyInfo privateKeyInfo = keypair.getPrivateKeyInfo();
            local = sha1(privateKeyInfo.getEncoded());
        }
        else
        {
            throw new IllegalArgumentException("Unknown fingerprint format " + Arrays.toString(awsFingerprint));
        }
        return Arrays.equals(local, awsFingerprint);
    }

    @VisibleForTesting
    static byte[] sha1(final byte[] encoded)
    {
        try
        {
            return MessageDigest.getInstance("SHA-1").digest(encoded);
        }
        catch (final NoSuchAlgorithmException e)
        {
            throw new IllegalArgumentException(e);
        }
    }
}
