package com.atlassian.aws.ec2;

import com.amazonaws.services.ec2.model.InstanceType;
import com.amazonaws.services.ec2.model.VirtualizationType;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;

import static com.atlassian.aws.ec2.InstanceFamilyData.currentGeneration;
import static com.atlassian.aws.ec2.InstanceFamilyData.currentGenerationWithPv;
import static com.atlassian.aws.ec2.InstanceFamilyData.familyNames;
import static com.atlassian.aws.ec2.InstanceFamilyData.oldGenerationWithHvm;
import static com.atlassian.aws.ec2.InstanceFamilyData.oldGenerationWithPv;

/**
 * <p>An enumeration of the instance types available in EC2.</p>
 */
public enum EC2InstanceType
{
    /* General Purpose Instances */

    M3Medium(InstanceType.M3Medium,
            0, 1,
            "3.75 GiB RAM, 1 vCPU, 3 EC2 Compute Units, 4 GB of SSD-based local instance storage",
            "Networking Performance: Moderate"),
    M3Large(InstanceType.M3Large,
            0, 1,
            "7.5 GiB RAM, 2 VCPUs, 6.5 EC2 Compute Units, 32 GB of SSD-based local instance storage",
            "Networking Performance: Moderate"),
    M3Xlarge(InstanceType.M3Xlarge,
            0, 2,
            "13 EC2 Compute Units (4 virtual cores with 3.25 EC2 Compute Units each)", "15 GiB RAM",
            "Networking Performance: High",
            "EBS-Optimized Available: No"),

    M32xlarge(InstanceType.M32xlarge,
            0, 2,
            "26 EC2 Compute Units (8 virtual cores with 3.25 EC2 Compute Units each)",
            "30 GiB RAM",
            "Networking Performance: High",
            "EBS-Optimized Available: No"
    ),

    M4Large(InstanceType.M4Large, 0, 0, "8 GiB RAM, 2 vCPUs, Network Bandwidth: Moderate"),
    M4Xlarge(InstanceType.M4Xlarge, 0, 0, "16 GiB RAM, 4 vCPUs, Network Bandwidth: High"),
    M42xlarge(InstanceType.M42xlarge, 0, 0, "32 GiB RAM, 8 vCPUs, Network Bandwidth: High"),
    M44xlarge(InstanceType.M44xlarge, 0, 0, "64 GiB RAM, 16 vCPUs, Network Bandwidth: High"),
    M410xlarge(InstanceType.M410xlarge, 0, 0, "160 GiB RAM, 40 vCPUs, Network Bandwidth: 10 Gbps"),
    M416xlarge(InstanceType.M416xlarge, 0, 0, "256 GiB RAM, 64 vCPUs, Network Bandwidth: 20 Gbps"),

    M5Large(InstanceType.M5Large, 0, 0, "2 vCPUs", "8 GiB RAM", "EBS Optimized", "EBS Bandwidth: Up to 2,120 Mbps"),
    M5Xlarge(InstanceType.M5Xlarge, 0, 0, "4 vCPUs", "16 GiB RAM", "EBS Optimized", "EBS Bandwidth: Up to 2,120 Mbps"),
    M52xlarge(InstanceType.M52xlarge, 0, 0, "8 vCPUs", "32 GiB RAM", "EBS Optimized", "EBS Bandwidth: Up to 2,120 Mbps"),
    M54xlarge(InstanceType.M54xlarge, 0, 0, "16 vCPUs", "64 GiB RAM", "EBS Optimized", "EBS Bandwidth: 2,210 Mbps"),
    M512xlarge(InstanceType.M512xlarge, 0, 0, "48 vCPUs", "192 GiB RAM", "EBS Optimized", "EBS Bandwidth: 5,000 Mbps"),
    M524xlarge(InstanceType.M524xlarge, 0, 0, "96 vCPUs", "384 GiB RAM", "EBS Optimized", "EBS Bandwidth: 10,000 Mbps"),

    M1_SMALL(InstanceType.M1Small,
            1, 0,
            "1 EC2 Compute Unit (1 virtual core with 1 EC2 Compute Unit)", "1.7 GiB RAM",
            "160 GB instance storage",
             "32-bit or 64-bit platform",
             "Networking Performance: Low",
             "EBS-Optimized Available: No"
    ),

    M1Medium(InstanceType.M1Medium,
            1, 0,
            "3.75 GiB RAM", "M1 Medium Instance",
            "2 EC2 Compute Unit (1 virtual core with 2 EC2 Compute Unit)",
            "410 GB instance storage",
            "32-bit or 64-bit platform",
            "Networking Performance: Moderate",
            "EBS-Optimized Available: No"
    ),

    M1_LARGE(InstanceType.M1Large,
            2, 0,
            "4 EC2 Compute Units (2 virtual cores with 2 EC2 Compute Units each)", "7.5 GiB RAM",
            "850 GB instance storage",
            "Networking Performance: Moderate",
            "EBS-Optimized Available: 500 Mbps"
    ),

    M1_XLARGE(InstanceType.M1Xlarge,
            4, 0,
            "15 GiB RAM", "M1 Extra Large Instance",
            "8 EC2 Compute Units (4 virtual cores with 2 EC2 Compute Units each)",
            "1,690 GB instance storage",
            "Networking Performance: High",
            "EBS-Optimized Available: 1000 Mbps"
    ),

    T1Micro(InstanceType.T1Micro,
            0, 0,
            "1 vCPU", "0.6131 GiB RAM", "32-bit or 64-bit platform"
    ),

    T2Nano(InstanceType.T2Nano,
            0, 0,
            "1 vCPU", "0.5 GiB RAM", "32-bit or 64-bit platform"
    ),
    T2Micro(InstanceType.T2Micro,
            0, 0,
            "1 vCPU", "1 GiB RAM", "32-bit or 64-bit platform"
    ),
    T2Small(InstanceType.T2Small,
            0, 0,
            "1 vCPU", "2 GiB RAM", "32-bit or 64-bit platform"
    ),
    T2Medium(InstanceType.T2Medium, 0, 0, "2 vCPUs", "4 GiB RAM"),

    T2Large(InstanceType.T2Large, 0, 0, "2 vCPUs", "8 GiB RAM"),
    T2Xlarge(InstanceType.T2Xlarge, 0, 0, "4 vCPUs", "16 GiB RAM"),
    T22xlarge(InstanceType.T22xlarge, 0, 0, "8 vCPUs", "32 GiB RAM"),


    /* Compute Optimised Instances */

    C3Large(InstanceType.C3Large,
            0, 2,
            "3.75 GiB RAM, 2 vCPUs, 7 EC2 Compute Units, 32 GB of SSD-based local instance storage"),
    C3Xlarge(InstanceType.C3Xlarge,
            0, 2,
            "7 GiB RAM, 4 vCPUs, 14 EC2 Compute Units, 80 GB of SSD-based local instance storage"),
    C32xlarge(InstanceType.C32xlarge,
            0, 2,
            "15 GiB RAM, 8 vCPUs, 28 EC2 Compute Units, 160 GB of SSD-based local instance storage"),
    C34xlarge(InstanceType.C34xlarge,
            0, 2,
            "30 GiB RAM, 16 vCPUs, 55 EC2 Compute Units, 320 GB of SSD-based local instance storage"),
    C38xlarge(InstanceType.C38xlarge,
            0, 2,
            "60 GiB RAM, 32 vCPUs, 108 EC2 Compute Units, 640 GB of SSD-based local instance storage"),

    C5large(InstanceType.C5Large, 0, 0, "2 vCPUs", "4 GiB RAM", "EBS Optimized Storage", "EBS Bandwidth: Up to 2,250 Mbps"),
    C5Xlarge(InstanceType.C5Xlarge, 0, 0, "4 vCPUs", "8 GiB RAM", "EBS Optmized Storage", "EBS Bandwidth: Up to 2,250 Mbps"),
    C52xlarge(InstanceType.C52xlarge, 0, 0, "8 vCPUs", "16 GiB RAM", "EBS Optimized Storage", "EBS Bandwidth: Up to 2,250 Mbps"),
    C54xlarge(InstanceType.C54xlarge, 0, 0, "16 vCPUs", "32 GiB RAM", "EBS Optimized Storage", "EBS Bandwidth: 2,250 Mbps"),
    C59xlarge(InstanceType.C59xlarge, 0, 0, "36 vCPUs", "72 GiB RAM", "EBS Optimized Storage", "EBS Bandwidth: 4,500 Mbps"),
    C518xlarge(InstanceType.C518xlarge, 0, 0, "72 vCPUs", "144 GiB RAM", "EBS Optimized Storage", "EBS Bandwidth: 9,000 Mbps"),


    C1_MEDIUM(InstanceType.C1Medium,
            1, 0,
            "5 EC2 Compute Units (2 virtual cores with 2.5 EC2 Compute Units each)",
            "1.7 GiB RAM",
            "350 GB of instance storage",
            "32-bit or 64-bit platform",
            "Networking Performance: Moderate",
            "EBS-Optimized Available: No"
    ),
    C1_XLARGE(InstanceType.C1Xlarge,
            4, 0,
            "20 EC2 Compute Units (8 virtual cores with 2.5 EC2 Compute Units each)", "7 GiB RAM",
            "1690 GB of instance storage",
            "Networking Performance: High",
            "EBS-Optimized Available: No"
    ),

    Cc28xlarge(InstanceType.Cc28xlarge,
            4, 0,
            "88 EC2 Compute Units (2 x Intel Xeon E5-2670, eight-core)", "60.5 GiB RAM",
            "3370 GB of instance storage",
            "Network Bandwidth: 10 Gbps)",
            "EBS-Optimized Available: No"),

    C4Large(InstanceType.C4Large, 0, 0, "3.75 GiB RAM, 2 vCPUs"),
    C4Xlarge(InstanceType.C4Xlarge, 0, 0, "7.5 GiB RAM, 4 vCPUs"),
    C42xlarge(InstanceType.C42xlarge, 0, 0, "15 GiB RAM, 8 vCPUs"),
    C44xlarge(InstanceType.C44xlarge, 0, 0, "30 GiB RAM, 16 vCPUs"),
    C48xlarge(InstanceType.C48xlarge, 0, 0, "60 GiB RAM, 36 vCPUs"),

    /* FPGA instances */

    F12xlarge(InstanceType.F12xlarge, 0, 1, "1 FPGA", "8 vCPUs", "122 GiB RAM"),
    F116xlarge(InstanceType.F116xlarge, 0, 4, "8 FPGA", "64 vCPUs", "976 GiB RAM"),

    /* GPU Instances */

    G22xlarge(InstanceType.G22xlarge,
            0, 1,
            "15 GiB memory, 26 EC2 Compute Units, 1 x NVIDIA GRID GPU (Kepler GK104), 60 GB of local instance storage"),

    G28xlarge(InstanceType.G28xlarge,
            0, 2,
            "60 GiB memory, 32 vCPUs, 4 x NVIDIA GPU, 240 GB of local instance storage"),

    G34xlarge(InstanceType.G34xlarge,
            0, 0,
            "122 GiB RAM", "16 vCPUs", "1 x NVIDIA GPU", "Network Bandwidth: <10 Gbps"),

    G38xlarge(InstanceType.G38xlarge,
            0, 0,
            "244 GiB RAM", "32 vCPUs", "2 x NVIDIA GPU", "Network Bandwidth: 10 Gbps"),


    H12xlarge(InstanceType.H12xlarge, 1, 0, "8 vCPUs", "32 GiB RAM", "Networking Performance: Up to 10 Gigabit", "1 x 2TB HDD"),
    H14xlarge(InstanceType.H14xlarge, 2, 0,  "16 vCPUs", "64 GiB RAM", "Networking Performance: Up to 10 Gigabit", "2 x 2TB HDD"),
    H18xlarge(InstanceType.H18xlarge, 4, 0, "32 vCPUs", "128 GiB RAM", "Networking Performance: 10 Gigabit", "4 x 2TB HDD"),
    H116xlarge(InstanceType.H116xlarge, 8, 0, "64 vCPUs", "256 GiB RAM", "Networking Performance: 25 Gigabit", "8 x 2TB HDD"),

    G316xlarge(InstanceType.G316xlarge,
            0, 0,
            "488 GiB RAM", "64 vCPUs", "4 x NVIDIA GPU", "Network Bandwidth: 20 Gbps"),

    Cg14xlarge(InstanceType.Cg14xlarge,
            2, 0,
            "33.5 EC2 Compute Units (2 x Intel Xeon X5570, quad-core)", "22 GiB RAM",
            "2 x NVIDIA Tesla Fermi M2050 GPUs",
            "1690 GB of instance storage",
            "Network Bandwidth: 10 Gbps",
            "EBS-Optimized Available: No"),

    P2Xlarge(InstanceType.P2Xlarge, 0, 0, "61 GiB RAM, 4 vCPUs, 1 GPU, Network Bandwidth: High"),
    P28xlarge(InstanceType.P28xlarge, 0, 0, "488 GiB RAM, 32 vCPUs, 8 GPU, Network Bandwidth: 10 Gbps"),
    P216xlarge(InstanceType.P216xlarge, 0, 0, "732 GiB RAM, 64 vCPUs, 16 GPU, Network Bandwidth: 20 Gbps"),

    P32xlarge(InstanceType.P32xlarge, 0, 0, "1 GPU", "16 GB GPU RAM", "8 vCPUs", "61 GB RAM", "Network Bandwidth: Up to 10Gbps", "EBS Bandwidth: 1.5Gbps"),
    P38xlarge(InstanceType.P38xlarge, 0, 0, "4 GPUs", "64 GB GPU RAM", "32 vCPUs", "244 GB RAM", "Network Bandwidth: 10Gbps", "EBS Bandwidth: 7Gbps"),
    P316xlarge(InstanceType.P316xlarge, 0, 0, "8 GPUs", "128 GB GPU RAM", "64 vCPUs", "488 GB RAM", "Network Bandwidth: 25Gbps", "EBS Bandwidth: 14Gbps"),

    /* New Memory Optimised Instances */
    R3Large(InstanceType.R3Large,
            0, 1,
            "2 vCPUs", "15 GiB RAM",
            "1 x 32 GB of SSD-based instance storage"
    ),
    R3XLarge(InstanceType.R3Xlarge,
            0, 1,
            "4 vCPUs", "30.5 GiB RAM",
            "1 x 80 GB of SSD-based instance storage"
    ),
    R32xlarge(InstanceType.R32xlarge,
            0, 1,
            "8 vCPUs", "61 GiB RAM",
            "1 x 160 GB of SSD-based instance storage"
    ),
    R34xlarge(InstanceType.R34xlarge,
            0, 1,
            "16 vCPUs", "122 GiB RAM",
            "1 x 320 GB of SSD-based instance storage"
    ),
    R38xlarge(InstanceType.R38xlarge,
            0, 2,
            "32 vCPUs", "244 GiB RAM",
            "2 x 320 GB of SSD-based instance storage",
            "Network Bandwidth: 10 Gbps"
    ),

    R4Large(InstanceType.R4Large, 0, 0,"2 vCPUs", "15.25 GiB RAM", "Network Bandwidth: <10 Gbps"),
    R4Xlarge(InstanceType.R4Xlarge, 0, 0,"4 vCPUs", "30.5 GiB RAM", "Network Bandwidth: <10 Gbps"),
    R42xlarge(InstanceType.R42xlarge, 0, 0,"8 vCPUs", "61 GiB RAM", "Network Bandwidth: <10 Gbps"),
    R44xlarge(InstanceType.R44xlarge, 0, 0,"16 vCPUs", "122 GiB RAM", "Network Bandwidth: <10 Gbps"),
    R48xlarge(InstanceType.R48xlarge, 0, 0,"32 vCPUs", "244 GiB RAM", "Network Bandwidth: 10 Gbps"),
    R416xlarge(InstanceType.R416xlarge, 0, 0,"64 vCPUs", "488 GiB RAM", "Network Bandwidth: 20 Gbps"),

    /* Old Memory Optimised Instances */

    M2Xlarge(InstanceType.M2Xlarge,
            1, 0,
            "6.5 EC2 Compute Units (2 virtual cores with 3.25 EC2 Compute Units each)", "17.1 GiB RAM",
            "420 GB of instance storage",
             "Networking Performance: Moderate",
             "EBS-Optimized Available: No"
    ),

    M22xlarge(InstanceType.M22xlarge,
            1, 0,
            "13 EC2 Compute Units (4 virtual cores with 3.25 EC2 Compute Units each)", "34.2 GiB RAM",
            "850 GB of instance storage",
              "Networking Performance: Moderate",
              "EBS-Optimized Available: No"),


    M24xlarge(InstanceType.M24xlarge,
            2, 0,
            "26 EC2 Compute Units (8 virtual cores with 3.25 EC2 Compute Units each)", "68.4 GiB RAM",
            "1690 GB of instance storage",
            "Networking Performance: High",
            "EBS-Optimized Available: 1000 Mbps"),

    Cr18xlarge(InstanceType.Cr18xlarge,
            0, 2,
            "88 EC2 Compute Units (2 x Intel Xeon E5-2670, eight-core. Intel Turbo, NUMA)", "244 GiB RAM",
            "240 GB of SSD instance storage",
            "Networking Performance: 10 Gbps",
            "EBS-Optimized Available: No"),

    /* Storage Optimised Instances */

    I2Xlarge(InstanceType.I2Xlarge,
            0, 1,
            "30.5 GiB RAM, 4 vCPUs, 14 EC2 Compute Units, 800 GB of SSD-based instance storage"),
    I22xlarge(InstanceType.I22xlarge,
            0, 2,
            "61 GiB RAM, 8 vCPUs, 27 EC2 Compute Units, 2 x 800 GB of SSD-based instance storage"),
    I24xlarge(InstanceType.I24xlarge,
            0, 4,
            "122 GiB RAM, 16 vCPUs, 53 EC2 Compute Units, 4 x 800 GB of SSD-based instance storage"),
    I28xlarge(InstanceType.I28xlarge,
            0, 8,
            "244 GiB RAM, 32 vCPUs, 104 EC2 Compute Units, 8 x 800 GB of SSD-based instance storage, Network Bandwidth: 10 Gbps"),

    I3Large(InstanceType.I3Large,
            0, 1,
            "15.25 GiB RAM", "2 vCPUs", "Network Bandwidth: <10 Gbps"),
    I3Xlarge(InstanceType.I3Xlarge,
            0, 1,
            "30.5 GiB RAM", "4 vCPUs", "Network Bandwidth: <10 Gbps"),
    I32xlarge(InstanceType.I32xlarge,
            0, 1,
            "61 GiB RAM", "8 vCPUs", "Network Bandwidth: <10 Gbps"),
    I34xlarge(InstanceType.I34xlarge,
            0, 2,
            "122 GiB RAM", "16 vCPUs", "Network Bandwidth: <10 Gbps"),
    I38xlarge(InstanceType.I38xlarge,
            0, 4,
            "244 GiB RAM", "32 vCPUs", "Network Bandwidth: 10 Gbps"),
    I316xlarge(InstanceType.I316xlarge,
            0, 8,
            "488 GiB RAM", "64 vCPUs", "Network Bandwidth: 20 Gbps"),

    D2Xlarge(InstanceType.D2Xlarge,
            3, 0,
            "14 EC2 Compute Units (4 virtual cores)", "30.5 GiB RAM",
            "Network Bandwidth: Moderate",
            "EBS-Optimized Available: By Default"),

    D22xlarge(InstanceType.D22xlarge,
            6, 0,
            "28 EC2 Compute Units (8 virtual cores)", "61 GiB RAM",
            "Network Bandwidth: High",
            "EBS-Optimized Available: By Default"),

    D24xlarge(InstanceType.D24xlarge,
            12, 0,
            "56 EC2 Compute Units (16 virtual cores)", "122 GiB RAM",
            "Network Bandwidth: High",
            "EBS-Optimized Available: By Default"),

    D28xlarge(InstanceType.D28xlarge,
            36, 0,
            "116 EC2 Compute Units (36 virtual cores)", "244 GiB RAM",
            "Network Bandwidth: 10 Gbps",
            "EBS-Optimized Available: By Default"),

    Hs18xlarge(InstanceType.Hs18xlarge,
            24, 0,
            "35 EC2 Compute Units (16 virtual cores)", "117 GiB RAM",
            "24 hard disk drives each with 2 TB of instance storage",
            "Network Bandwidth: 10 Gbps",
            "Storage I/O Performance: Very High",
            "EBS-Optimized Available: No"),

    Hi14xlarge(InstanceType.Hi14xlarge,
            0, 2,
            "35 EC2 Compute Units (16 virtual cores)", "60.5 GiB RAM",
            "2 SSD-based volumes each with 1024 GB of instance storage",
            "Network Bandwidth: 10 Gbps",
            "Storage I/O Performance: Very High",
            "EBS-Optimized Available: No"),

    X116xlarge(InstanceType.X116xlarge,
            0, 2,
            "? vCPUs", "? GiB RAM",
            "? GB of SSD-based instance storage",
            "Network Bandwidth: 10 Gbps Ethernet",
            "EBS-Optimized Available: By Default"),
    X132xlarge(InstanceType.X132xlarge,
            0, 2,
            "128 vCPUs", "1952 GiB RAM",
            "2 x 1920 GB of SSD-based instance storage",
            "Network Bandwidth: 10 Gbps Ethernet",
            "EBS-Optimized Available: By Default"),

    X1eXlarge(InstanceType.X1eXlarge, 0, 1, "4 vCPUs", "122 GiB", "1 x 120 GB SSD", "EBS Bandwidth: 500 Mbps"),
    X1e2xlarge(InstanceType.X1e2xlarge, 0, 1, "8 vCPUs", "244 GiB", "1 x 240 GB SSD", "EBS Bandwidth: 1,000 Mbps"),
    X1e4xlarge(InstanceType.X1e4xlarge, 0, 1, "16 vCPUs", "488 GiB", "1 x 480 GB SSD", "EBS Bandwidth: 1,750 Mbps"),
    X1e8xlarge(InstanceType.X1e8xlarge, 0, 1, "32 vCPUs", "976 GiB", "1 x 960 GB SSD", "EBS Bandwidth: 3,500 Mbps"),
    X1e16xlarge(InstanceType.X1e16xlarge, 0, 1, "64 vCPUs", "1,952 GiB", "1 x 1,920 GB SSD", "EBS Bandwidth: 7,000 Mbps"),
    X1e32xlarge(InstanceType.X1e32xlarge, 0, 2, "128 vCPUs", "3,904 GiB", "2 x 1,920 GB SSD", "EBS Bandwidth: 14,000 Mbps"),
    ;

    public static final EC2InstanceType DEFAULT = M1_SMALL;

    private final String name;
    private final int ssdCnt;
    private final int magneticDiscCnt;

    private String description;

    private final InstanceType apiParameterValue;

    @VisibleForTesting
    final String family;

    EC2InstanceType(final InstanceType apiParameterValue, final int magneticDiscCnt, final int ssdCnt, final String... descriptionLines)
    {
        this.name = generateName(apiParameterValue);
        this.ssdCnt = ssdCnt;
        this.magneticDiscCnt = magneticDiscCnt;
        if (descriptionLines.length>1)
        {
            this.description = Joiner.on('\n').join(descriptionLines);
        }
        else
        {
            this.description = StringUtils.replace(descriptionLines[0], ",", "\n");
        }

        if (ssdCnt == 0 && magneticDiscCnt == 0) {
            description = description + "\nEBS only";
        }
        this.apiParameterValue = apiParameterValue;
        this.family = StringUtils.substringBefore(apiParameterValue.toString(), ".").toUpperCase();
    }

    @VisibleForTesting
    static String generateName(final InstanceType apiParameterValue) {
        final String[] familyIdAndType = apiParameterValue.toString().split("\\.");
        final String familyId = familyIdAndType[0].toUpperCase();
        final String familyName = familyNames.get(familyId);
        Preconditions.checkNotNull(familyName, "Unknown family " + familyId);
        final String s = familyIdAndType[1]
                .replaceFirst("xlarge", "X Large");
        return familyId + " " + familyName + " " + StringUtils.capitalize(s);
    }

    public String getName()
    {
        return name;
    }

    public String getDescription()
    {
        return description;
    }

    public int getSsdCnt()
    {
        return ssdCnt;
    }

    public InstanceType getAwsInstanceType()
    {
        return apiParameterValue;
    }

    public boolean supports(final VirtualizationType virtualisation)
    {
        switch (virtualisation)
        {
            case Hvm:
                return currentGeneration.contains(family) || oldGenerationWithHvm.contains(family);
            case Paravirtual:
                return currentGenerationWithPv.contains(family) || oldGenerationWithPv.contains(family);
            default:
                throw new UnsupportedOperationException("Unknown virtualisation: " + virtualisation);
        }
    }

    public static EC2InstanceType from(final InstanceType instanceType)
    {
        Preconditions.checkNotNull(instanceType);
        for (final EC2InstanceType ec2InstanceType : values())
        {
            if (ec2InstanceType.getAwsInstanceType() == instanceType)
            {
                return ec2InstanceType;
            }
        }
        throw new IllegalArgumentException("Unknown instance type: " + instanceType);
    }

    @Override
    public String toString()
    {
        return name;
    }
}
