package com.atlassian.aws.ec2;

import com.atlassian.aws.AWSException;
import com.atlassian.aws.ec2.awssdk.AwsSupportConstants;
import com.atlassian.aws.ec2.caches.Ec2CacheMissException;
import org.apache.log4j.Logger;

public enum EC2InstanceState
{
    INITIAL,
    BIDDING
            {
                @Override
                void supervise(final RemoteEC2InstanceImpl impl) throws AWSException
                {
                    final InstanceReservationDescription instance = impl.describeInstance();

                    handleState(AwsSupportConstants.InstanceStateName.SpotOpen,
                                instance, impl,
                                AwsSupportConstants.InstanceStateName.SpotClosed,
                                AwsSupportConstants.InstanceStateName.SpotCancelled,
                                AwsSupportConstants.InstanceStateName.SpotFailed,
                                AwsSupportConstants.InstanceStateName.SpotActive);

                    if (impl.getInstanceStatus().isDeadlinePassed())
                    {
                        throw new AWSException("Deadline passed waiting for spot request " + impl.getInstanceStatus().getSensibleId() + " to be realised.");
                    }
                }

                @Override
                void supervisionFailure(final RemoteEC2InstanceImpl impl, final Throwable throwable)
                {
                    final String msg = "Spot request " + impl.getInstanceStatus().getSensibleId() + " has not been realised.  Attempting to terminate.";
                    terminate(impl, throwable, msg);
                    impl.setSupervisionState(FAILED_TO_START, msg, throwable);
                }
            },
    PENDING
            {
                @Override
                void supervise(final RemoteEC2InstanceImpl impl) throws AWSException
                {
                    final InstanceReservationDescription instance = impl.describeInstance();
                    handleState(AwsSupportConstants.InstanceStateName.Pending,
                                instance, impl,
                                AwsSupportConstants.InstanceStateName.Running,
                                AwsSupportConstants.InstanceStateName.Stopping,
                                AwsSupportConstants.InstanceStateName.Stopped,
                                AwsSupportConstants.InstanceStateName.ShuttingDown,
                                AwsSupportConstants.InstanceStateName.Terminated);

                    if (impl.getInstanceStatus().isDeadlinePassed())
                    {
                        throw new AWSException("Deadline passed waiting for EC2 instance " + impl.getInstanceId() + " to start.");
                    }
                }

                @Override
                void supervisionFailure(final RemoteEC2InstanceImpl impl, final Throwable throwable)
                {
                    final String msg = "EC2 instance " + impl.getInstanceId() + " failed to start.  Attempting to terminate.";
                    terminate(impl, throwable, msg);
                    impl.setSupervisionState(FAILED_TO_START, msg, throwable);
                }

            },
    FAILED_TO_START,
    RUNNING
            {
                @Override
                void supervise(final RemoteEC2InstanceImpl impl) throws AWSException
                {
                    final InstanceReservationDescription instance;
                    try
                    {
                        instance = impl.describeInstance();
                        handleState(AwsSupportConstants.InstanceStateName.Running,
                                    instance, impl, AwsSupportConstants.InstanceStateName.Stopping,
                                    AwsSupportConstants.InstanceStateName.Stopped,
                                    AwsSupportConstants.InstanceStateName.ShuttingDown,
                                    AwsSupportConstants.InstanceStateName.Terminated);
                    }
                    catch (final Ec2CacheMissException e)
                    {
                        final String msg = "Instance " + impl.getInstanceId() + " is gone, assuming it's been terminated";
                        log.warn(msg);
                        impl.getInstanceStatus().setState(AwsSupportConstants.InstanceStateName.Terminated);
                        impl.setSupervisionState(EC2InstanceState.TERMINATED, msg, null);
                    }
                }

                @Override
                void supervisionFailure(final RemoteEC2InstanceImpl impl, final Throwable throwable)
                {
                    final String details = "Failure in EC2 instance " + impl.getInstanceId() + ".  Attempting to terminate.";
                    terminate(impl, throwable, details);
                    impl.setSupervisionState(UNKNOWN, details, throwable);
                }

            },
    STOPPING
            {
                @Override
                void supervise(final RemoteEC2InstanceImpl impl) throws AWSException
                {
                    final InstanceReservationDescription instance = impl.describeInstance();

                    handleState(AwsSupportConstants.InstanceStateName.Stopping,
                            instance, impl, AwsSupportConstants.InstanceStateName.Stopped,
                                AwsSupportConstants.InstanceStateName.Running,
                                AwsSupportConstants.InstanceStateName.ShuttingDown,
                                AwsSupportConstants.InstanceStateName.Terminated);
                }

                @Override
                void supervisionFailure(final RemoteEC2InstanceImpl impl, final Throwable throwable)
                {
                    final String details = "Failure in EC2 instance " + impl.getInstanceId() + ".";
                    log.error(details, throwable);
                    impl.setSupervisionState(UNKNOWN, details, throwable);
                }
            },
    STOPPED
            {
                @Override
                void supervise(final RemoteEC2InstanceImpl impl) throws AWSException
                {
                    final InstanceReservationDescription instance = impl.describeInstance();

                    handleState(AwsSupportConstants.InstanceStateName.Stopped,
                            instance, impl, AwsSupportConstants.InstanceStateName.Stopping,
                                AwsSupportConstants.InstanceStateName.Running,
                                AwsSupportConstants.InstanceStateName.ShuttingDown,
                                AwsSupportConstants.InstanceStateName.Terminated);
                }

                @Override
                void supervisionFailure(final RemoteEC2InstanceImpl impl, final Throwable throwable)
                {
                    final String details = "Failure in EC2 instance " + impl.getInstanceId() + ".";
                    log.error(details, throwable);
                    impl.setSupervisionState(UNKNOWN, details, throwable);
                }
            },
    SHUTTING_DOWN
            {
                @Override
                void supervise(final RemoteEC2InstanceImpl impl) throws AWSException
                {
                    final InstanceReservationDescription instance = impl.describeInstance();

                    handleState(AwsSupportConstants.InstanceStateName.ShuttingDown,
                            instance, impl, AwsSupportConstants.InstanceStateName.Terminated);
                }

                @Override
                void supervisionFailure(final RemoteEC2InstanceImpl impl, final Throwable throwable)
                {
                    final String details = "EC2 instance " + impl.getInstanceId() + " failed to shut down.";
                    log.error(details, throwable);
                    impl.setSupervisionState(UNKNOWN, details, throwable);
                }

            },
    TERMINATED,
    UNKNOWN;

    private static void terminate(final RemoteEC2InstanceImpl impl, final Throwable throwable, final String details)
    {
        log.error(details, throwable);
        try
        {
            impl.asyncTerminate();
        }
        catch (final Throwable throwable2)
        {
            log.error("Failed to terminate EC2 instance " + impl.getInstanceId(), throwable2);
        }
    }

    private static void handleState(final AwsSupportConstants.InstanceStateName previousState,
                                    final InstanceReservationDescription instance, final RemoteEC2InstanceImpl handler,
                                    final AwsSupportConstants.InstanceStateName ... validStates) throws AWSException
    {
        AwsSupportConstants.InstanceStateName currentState;
        try
        {
            currentState = instance.getState();
        }
        catch (IllegalArgumentException e)
        {
            throw handler.unexpectedStateException(instance);
        }

        if (currentState == previousState)
        {
            if (currentState==AwsSupportConstants.InstanceStateName.Running)
            {
                final String oldAddress = handler.getInstanceStatus().getAddress();
                final String currentAddress = instance.getAddress();
                if (!currentAddress.equals(oldAddress))
                {
                    handler.handleAddressChange(instance);
                }
            }
            return;
        }
        
        for (AwsSupportConstants.InstanceStateName validState : validStates)
        {
            if (currentState == validState )
            {
                handler.handleStateChange(instance, currentState);
                return;
            }
        }
        throw handler.unexpectedStateException(instance);
    }

    private static final org.apache.log4j.Logger log = Logger.getLogger(EC2InstanceState.class);

    final boolean isFinal()
    {
        return this==FAILED_TO_START || this==TERMINATED || this==UNKNOWN;
    }

    void supervise(final RemoteEC2InstanceImpl impl) throws AWSException
    {
        throw new UnsupportedOperationException(this + " is not implemented.");
    }

    void supervisionFailure(final RemoteEC2InstanceImpl impl, final Throwable throwable)
    {
        throw new UnsupportedOperationException(this + " is not implemented.");
    }

}
