package com.atlassian.aws.ec2;

import com.atlassian.aws.ec2.model.AvailabilityZoneId;
import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;

public class AvailabilityZoneChooser
{
    private static final Random RANDOM = new Random();

    private final LoadingCache<EC2InstanceType, Map<AvailabilityZoneId, DateTime>> azCapacityErrors = CacheBuilder.newBuilder().build(new CacheLoader<EC2InstanceType, Map<AvailabilityZoneId, DateTime>>()
    {
        @Override
        public Map<AvailabilityZoneId, DateTime> load(@NotNull final EC2InstanceType key) throws Exception
        {
            return new HashMap<>();
        }
    });

    @NotNull
    public AvailabilityZoneId choose(@NotNull final Iterable<AvailabilityZoneId> availabilityZones, @NotNull final EC2InstanceType instanceType)
    {
        Preconditions.checkArgument(!Iterables.isEmpty(availabilityZones), "No availablility zones to choose from");
        final Map<AvailabilityZoneId, DateTime> azsWithErrors = azCapacityErrors.getUnchecked(instanceType);

        final Set<AvailabilityZoneId> goodAvailabilityZones = Sets.newHashSet(availabilityZones);
        final List<AvailabilityZoneId> worseAvailabilityZones = new ArrayList<>();

        final DateTime oneHourAgo = new DateTime().minusHours(1);
        for (final AvailabilityZoneId availabilityZone : availabilityZones)
        {
            final DateTime errorTimestamp = azsWithErrors.get(availabilityZone);
            if (errorTimestamp==null)
            {
                continue;
            }

            if (errorTimestamp.isAfter(oneHourAgo))
            {
                goodAvailabilityZones.remove(availabilityZone);
                worseAvailabilityZones.add(availabilityZone);
            }
        }

        if (!goodAvailabilityZones.isEmpty())
        {
            return pickRandomElement(goodAvailabilityZones);
        }
        return pickOldestElement(worseAvailabilityZones, azsWithErrors);
    }

    private AvailabilityZoneId pickRandomElement(final Collection<AvailabilityZoneId> collection)
    {
        return Iterables.get(collection, RANDOM.nextInt(collection.size()));
    }

    private AvailabilityZoneId pickOldestElement(final Iterable<AvailabilityZoneId> interestingZones, final Map<AvailabilityZoneId, DateTime> azsWithErrors)
    {
        DateTime oldestErrorTimestamp = null;
        AvailabilityZoneId oldestFullAvailabilityZone = null;
        for (final AvailabilityZoneId availabilityZone : interestingZones)
        {
            final DateTime errorTimestamp = azsWithErrors.get(availabilityZone);
            if (errorTimestamp==null)
            {
                continue;//should not happen
            }
            if (oldestErrorTimestamp==null || errorTimestamp.isBefore(oldestErrorTimestamp))
            {
                oldestErrorTimestamp = errorTimestamp;
                oldestFullAvailabilityZone=availabilityZone;
            }
        }
        assert oldestFullAvailabilityZone!=null;
        return oldestFullAvailabilityZone;
    }

    public void blacklist(@NotNull final AvailabilityZoneId availabilityZone, @NotNull final EC2InstanceType instanceType)
    {
        final Map<AvailabilityZoneId, DateTime> availabilityZones = azCapacityErrors.getUnchecked(instanceType);
        availabilityZones.put(availabilityZone, new DateTime());
    }
}
