package com.atlassian.aws;

import com.amazonaws.ClientConfiguration;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.services.ec2.AmazonEC2Async;
import com.amazonaws.services.ec2.AmazonEC2AsyncClient;
import com.atlassian.aws.ec2.awssdk.AwsSupportConstants;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.time.Duration;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

public class Ec2ClientFactoryImpl implements Ec2ClientFactory
{
    private static final String HTTPS = "https";
    private static final String HTTP = "http";

    private static final int DEFAULT_TRANSMISSION_TIMEOUT_SECONDS = 50;
    private static final int DEFAULT_MAX_RETRIES = 5;
    private final String endpoint;

    private static final String EC2_TRANSMISSION_TIMEOUT_PROPERTY = "atlassian.ec2.transmission.timeout";
    private Duration transmissionTimeout;

    private int maxRetries = DEFAULT_MAX_RETRIES;

    public Ec2ClientFactoryImpl(@Nullable String endpoint)
    {
        this.endpoint = endpoint;
        Integer transmissionTimeoutSeconds = Integer.getInteger(EC2_TRANSMISSION_TIMEOUT_PROPERTY, DEFAULT_TRANSMISSION_TIMEOUT_SECONDS);
        transmissionTimeout = Duration.ofSeconds(transmissionTimeoutSeconds);
    }

    public Ec2ClientFactoryImpl()
    {
        this(null);
    }

    @Override
    @NotNull
    public AmazonEC2Async newAmazonEc2Async(@NotNull final AwsSupportConstants.Region region, @NotNull final AWSCredentials awsCredentials, @NotNull final ScheduledExecutorService scheduledExecutorService)
    {
        return CallTimingProxy.wrap(newAwsAsyncClient(region, awsCredentials, scheduledExecutorService), AmazonEC2Async.class);
    }

    @Override
    @NotNull
    public AmazonEC2AsyncClient newAwsAsyncClient(@NotNull final AwsSupportConstants.Region region, @NotNull final AWSCredentials awsCredentials, @NotNull final ScheduledExecutorService scheduledExecutorService)
    {
        final String enpointToUse;
        if (region != AwsSupportConstants.Region.CUSTOM)
        {
            enpointToUse = region.getEndpoint();
        }
        else
        {
            if (endpoint==null)
            {
                throw new IllegalArgumentException("No custom endpoint available");
            }
            enpointToUse = endpoint;
        }

        return newAwsAsyncClient(enpointToUse, awsCredentials, scheduledExecutorService);
    }

    private AmazonEC2AsyncClient newAwsAsyncClient(@NotNull final String endpoint, final AWSCredentials awsCredentials, final ScheduledExecutorService scheduledExecutorService)
    {
        final int transmissionTimeoutMs = (int)transmissionTimeout.toMillis();

        final ClientConfiguration clientConfiguration = HttpClientConfigurationUtilsDoNotUse.newClientConfiguration(transmissionTimeoutMs, maxRetries);
        final AmazonEC2AsyncClient amazonEC2AsyncClient = new AmazonEC2AsyncClient(awsCredentials, clientConfiguration, scheduledExecutorService);
        amazonEC2AsyncClient.setEndpoint(endpoint);

        return amazonEC2AsyncClient;
    }

    /**
     * Sets the timeout used for connection to host and transmission of data. 
     *
     * @param transmissionTimeoutSeconds timeout in seconds
     */
    public void setTransmissionTimeout(final int transmissionTimeoutSeconds)
    {
        this.transmissionTimeout = Duration.ofSeconds(transmissionTimeoutSeconds);
    }

    /**
     * Sets maximum retries before the connection is considered dead.
     * Defaults to five if not set.
     *
     * @param maxRetries the amount of retries
     */
    public void setMaxRetries(final int maxRetries)
    {
        this.maxRetries = maxRetries;
    }
}
