package com.atlassian.aws;

import com.atlassian.aws.ec2.model.ResourceId;
import com.atlassian.aws.utils.AwsSuppliers;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import org.apache.log4j.Logger;
import org.jetbrains.annotations.NotNull;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

public abstract class AwsOmeCache<T>
{
    private static final Duration DEFAULT_MAXIMUM_DATA_AGE = Duration.ofSeconds(10);

    private volatile Supplier<Map<String, T>> recentData;
    private final Logger log;
    private final Duration maximumDataAge;

    public AwsOmeCache(final Logger log, final Duration maximumDataAge)
    {
        this.log = log;
        this.maximumDataAge = maximumDataAge;
        recentData = newRecentDataSupplier(maximumDataAge);
    }

    private Supplier<Map<String, T>> newRecentDataSupplier(final Duration maximumDataAge)
    {
        return AwsSuppliers.memoizeWithAdaptiveTtl(
                new Supplier<Map<String, T>>()
                {
                    @Override
                    public Map<String, T> get()
                    {
                        log.debug("existing data is stale, requesting new data");

                        List<T> resources = refreshAllData();

                        Map<String, T> resourceId2resourceMap = new HashMap<>();


                        for (T resource : resources)
                        {
                            resourceId2resourceMap.put(toResourceId(resource), resource);
                        }

                        return Collections.unmodifiableMap(resourceId2resourceMap);
                    }
                },
                maximumDataAge, maximumDataAge.multipliedBy(3));
    }

    public AwsOmeCache(final Logger log)
    {
        this(log, DEFAULT_MAXIMUM_DATA_AGE);
    }

    private Collection<T> filterResources(Map<String, T> resourceId2resource, final String[] resourceIds, boolean isDataPotentiallyStale)
    {
        Collection<T> result = new ArrayList<>();
        for (final String resourceId : Sets.newLinkedHashSet(Arrays.asList(resourceIds)))
        {
            T resource = resourceId2resource.get(resourceId);

            if (resource != null)
            {
                result.add(resource);
            }
            else
            {
                if (isDataPotentiallyStale)
                {
                    log.debug("cache miss on [" + resourceId + "]");
                    recentData = newRecentDataSupplier(maximumDataAge);
                    return filterResources(recentData.get(), resourceIds, false);
                }
                else
                {
                    log.info("query for a non-existing resource [" + resourceId + "]");
                    onResourceLookupFailure(resourceId);
                }
            }
        }
        return result;
    }

    /**
     * Returns a recently retrieved snapshot of resources of given type, optionally limiting them to the supplied
     * list.
     *
     * @return the list of all resources of given type on this account.
     */
    @NotNull
    public Collection<T> describe(final String... resourceIds)
    {
        final boolean noFiltering = resourceIds.length == 0;

        log.debug("describing resources: " + (noFiltering ? "ALL" : Arrays.toString(resourceIds)));

        Map<String, T> resourceId2ResourceMap = recentData.get();

        Collection<T> resources = noFiltering ?
                                  resourceId2ResourceMap.values():
                                  filterResources(resourceId2ResourceMap, resourceIds, true);
        return resources;
    }

    @NotNull
    public Collection<T> describe(final Iterable<String> resourceIds)
    {
        return describe(Iterables.toArray(resourceIds, String.class));
    }

    @NotNull
    public Collection<T> describeResources(final Iterable<? extends ResourceId<?>> resourceIds)
    {
        return describe(Iterables.toArray(ResourceId.getIds(resourceIds), String.class));
    }

    public void setMaximumStatusAgeSeconds(final int maximumStatusAgeSeconds)
    {
        recentData = newRecentDataSupplier(Duration.ofSeconds(maximumStatusAgeSeconds));
    }

    protected abstract List<T> refreshAllData();

    protected abstract String toResourceId(T resource);

    protected abstract void onResourceLookupFailure(final String resourceId);
}
