package com.atlassian.aws;

import com.amazonaws.services.ec2.AmazonEC2Async;
import com.amazonaws.services.ec2.AmazonEC2AsyncClient;
import com.amazonaws.services.ec2.model.AccountAttribute;
import com.amazonaws.services.ec2.model.Address;
import com.amazonaws.services.ec2.model.AvailabilityZone;
import com.amazonaws.services.ec2.model.Image;
import com.amazonaws.services.ec2.model.Instance;
import com.amazonaws.services.ec2.model.KeyPair;
import com.amazonaws.services.ec2.model.KeyPairInfo;
import com.amazonaws.services.ec2.model.SecurityGroup;
import com.amazonaws.services.ec2.model.SpotInstanceRequest;
import com.amazonaws.services.ec2.model.Subnet;
import com.amazonaws.services.ec2.model.Volume;
import com.amazonaws.services.ec2.model.Vpc;
import com.atlassian.aws.ec2.EC2InstanceListener;
import com.atlassian.aws.ec2.InstanceLaunchConfiguration;
import com.atlassian.aws.ec2.Protocol;
import com.atlassian.aws.ec2.RemoteEC2Instance;
import com.atlassian.aws.ec2.SpotPrices;
import com.atlassian.aws.ec2.awssdk.AwsSupportConstants;
import com.atlassian.aws.ec2.caches.SubnetCache;
import com.atlassian.aws.ec2.model.InstanceId;
import com.atlassian.aws.ec2.model.VpcId;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.util.Collection;
import java.util.List;
import java.util.Map;

public interface AWSAccount
{

    /**
     * <p>Checks whether the credentials encapsulated by this object are valid.</p>
     * 
     * @return <code>null</code> if the credentials are valid; <code>String</code> with error message otherwise.
     * @throws AWSException if a problem prevents the validity of the credentials from being determined.
     */
    @Nullable
    String getAccountValidationError() throws AWSException;

    /**
     * <p>Place an order for a new EC2 instance.</p>
     *
     * @return An object representing the EC2 instance.
     */
    @NotNull
    RemoteEC2Instance newEC2Instance(@NotNull final InstanceLaunchConfiguration instanceLaunchConfiguration, EC2InstanceListener listener);

    @NotNull
    Iterable<SecurityGroup> describeSecurityGroups() throws AWSException;

    @NotNull
    SecurityGroup newSecurityGroup(@NotNull String name, @NotNull String description, @Nullable VpcId vpcId) throws AWSException;

    void ensureInboundTrafficIsAllowed(SecurityGroup group, Protocol protocol, String cidrIpRange, int port);

    @NotNull
    Map<String, KeyPairInfo> describeEc2KeyPairs(final String... keyNames);

    @NotNull
    KeyPair newEC2KeyPair(String name) throws AWSException;

    /**
     * Returns a list of availability zones and their status.
     *
     * @return A map of zones and their availability keyed by zone name.
     * @throws AWSException When there's an error querying EC2 service
     */
    @NotNull
    Map<String, AvailabilityZone> getAvailabilityZones() throws AWSException;

    /**
     * Returns the console logs returned from EC2. Note that logs are
     * not returned until sometime after the agent starts
     *
     * @param instanceId
     * @return null if the console is empty or an error occurred while retrieving it.
     */
    @Nullable
    String getConsoleOutput(String instanceId);
    

    /**
     * Returns the list of all instances on this account that are currently up.
     * @return the list of all instances on this account that are currently up.
     */
    @NotNull
    Collection<Instance> getAllInstances() throws AWSException;

    /**
     * Returns a recently retrieved snapshot of pending ('open') spot instance requests
     * optionally limiting them to the supplied list.
     *
     * @return the list of all pending spot instance requests on this account.
     */
    @NotNull
    Collection<SpotInstanceRequest> describePendingSpotInstanceRequests(String... spotInstanceRequestId);

    /**
     * @return a recently retrieved list of instances on this account. Does not return terminated instances.
     */
    Collection<Instance> describeInstances(String... instanceIds);


    /**
     * @deprecated since 5.10 use {@link #getPassword(File, InstanceId)} instead
     */
    @Deprecated
    String getPassword(@NotNull final String privateKeyFile, @NotNull final String instanceIds);
    /**
     * Retrieves the generated password for instance.
     *
     * @return null for instances without password randomisation or for instances on which the password hasn't been generated yet. Otherwise returns the password to the Administrator account.
     *
     */
    @Nullable
    String getPassword(@NotNull final File privateKeyFile, @NotNull final InstanceId instanceId);

    /**
     * @return a recently retrieved image
     */
    Image describeImage(String imageId);

    /**
     * @return a recently retrieved list of images on this account.
     */
    List<Image> describeImages(String... imageIds);

    /**
     * @return a recently retrieved list of all spot instance requests on this account.
     */
    Collection<SpotInstanceRequest> describeSpotInstanceRequests(String... spotInstanceRequestIds);
    /**
     * Returns the list of all ebs volumes on this account.
     * @return a recently retrieved list of all ebs volumes on this account.
     * @throws AWSException when there is an error querying EC2 service.
     */
    @NotNull
    Collection<Volume> describeVolumes() throws AWSException;

    String createVolume(@NotNull final String ebsSnapshotId, @NotNull final String availabilityZone);

    void attachVolume(@NotNull String volumeId, @NotNull String instanceId, @NotNull String device, final boolean deleteOnTermination);

    /**
     * Shutdown the elastic instance with the given id.
     * @param instanceId
     * @throws AWSException when there is an error shutting down EC2 instance.
     */
    void shutdownInstance(String instanceId) throws AWSException;

    /**
     * Delete the volume with the given id.
     * @param volumeId
     * @throws AWSException when there is an error deleting the volume.
     */
    void deleteVolume(String volumeId) throws AWSException;

    @NotNull
    SpotPrices getSpotPrices();

    /**
     * @deprecated since 5.9 use {@link #getAmazonEc2()} instead
     * @return
     */
    @Deprecated
    AmazonEC2AsyncClient getAwsClient();

    AmazonEC2Async getAmazonEc2();

    void setMaximumEbsVolumeStatusAgeSeconds(int maximumStatusAgeSeconds);

    void setMaximumInstanceStatusAgeSeconds(int maximumStatusAgeSeconds);

    void setMaximumSpotRequestStatusAgeSeconds(int maximumStatusAgeSeconds);

    /**
     * @return a map of available VPCs and subnets defined in these VPCs
     * @throws AWSException
     */
    @NotNull
    Map<Vpc, Collection<Subnet>> describeVpcs() throws AWSException;

    /**
     * Returns VPC subnets currently defined in EC2.
     * @param subnetIds  No parameters = all subnets. Otherwise, subnet ids that should be returned.
     */
    @NotNull
    Collection<Subnet> describeSubnets(final String... subnetIds);

    @NotNull
    List<Address> describeAddresses(final String... domains);

    void createTag(@NotNull String resourceId, @NotNull String key, @NotNull String value);

    AwsSupportConstants.Region getRegion();

    void cancelSpotInstanceRequests(String... spotInstanceRequestIds);

    @NotNull
    Map<String,AccountAttribute> getAccountAttributes();

    SubnetCache getSubnetCache();
}
