package com.atlassian.aws;

import com.amazonaws.AmazonClientException;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.regions.Regions;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.Region;
import com.atlassian.aws.s3.BambooAmazonS3Client;
import com.atlassian.aws.s3.S3Support;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.jetbrains.annotations.Nullable;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

public class AmazonClients
{
    private static final int DEFAULT_TRANSMISSION_TIMEOUT_MS = (int)TimeUnit.SECONDS.toMillis(30);
    private static final int DEFAULT_MAX_RETRIES = 5;

    private static final Logger log = Logger.getLogger(AmazonClients.class);
    private static final AtomicBoolean logRegionOnce = new AtomicBoolean();

    private static AtomicReference<com.amazonaws.regions.Region> currentRegion = null;

    private AmazonClients()
    {
    }

    public static AmazonS3ClientBuilder newS3Client(final AWSCredentials awsCredentials) {
        final ClientConfiguration clientConfiguration = HttpClientConfigurationUtilsDoNotUse.newClientConfiguration(DEFAULT_TRANSMISSION_TIMEOUT_MS, DEFAULT_MAX_RETRIES);
        
        final AmazonS3ClientBuilder s3ClientBuilder = AmazonS3ClientBuilder.standard()
                .withDualstackEnabled(true)
                .withCredentials(new AWSStaticCredentialsProvider(awsCredentials))
                .withClientConfiguration(clientConfiguration);

        final com.amazonaws.regions.Region currentRegion = getCurrentRegion();
        if (currentRegion!=null) {
            final Level level = logRegionOnce.compareAndSet(false, true) ? Level.INFO : Level.DEBUG;
            log.log(level, "Detected current region as " + currentRegion + ", using it for S3 access");
            s3ClientBuilder.withRegion(toRegions(currentRegion));
        }
        
        return s3ClientBuilder;
    }

    @Deprecated
    public static BambooAmazonS3Client newAmazonS3Client(final AWSCredentials awsCredentials)
    {
        final ClientConfiguration clientConfiguration = HttpClientConfigurationUtilsDoNotUse.newClientConfiguration(DEFAULT_TRANSMISSION_TIMEOUT_MS, DEFAULT_MAX_RETRIES);
        final BambooAmazonS3Client amazonS3Client = new BambooAmazonS3Client(awsCredentials, clientConfiguration);

        final com.amazonaws.regions.Region currentRegion = getCurrentRegion();
        if (currentRegion!=null)
        {
            final Level level = logRegionOnce.compareAndSet(false, true) ? Level.INFO : Level.DEBUG;
            log.log(level, "Detected current region " + currentRegion + ", using it for S3 access");
            amazonS3Client.setRegion(currentRegion);
        }
        return amazonS3Client;
    }

    @Nullable
    private static synchronized com.amazonaws.regions.Region getCurrentRegion() {
        if (currentRegion==null) {
            currentRegion = new AtomicReference<>(Regions.getCurrentRegion());
        }
        return currentRegion.get();
    }

    public static AmazonS3ClientBuilder setBestEndpointForBucket(final AmazonS3ClientBuilder s3ClientBuilder, final String bucket) {
        log.info("Detecting bucket location for [" + bucket + "] via " + s3ClientBuilder.getRegion() + " endpoint");
        final String bucketLocation = getBucketLocation(s3ClientBuilder.build(), bucket);
        if (bucketLocation==null) {
            return s3ClientBuilder;
        }

        try {
            log.debug("Bucket location: " + bucketLocation);
            final com.amazonaws.regions.Region regionFromLocation = Region.fromValue(bucketLocation).toAWSRegion();
            final String oldRegion = s3ClientBuilder.getRegion();
            s3ClientBuilder.withRegion(Regions.fromName(regionFromLocation.getName()));
            if (getBucketLocation(s3ClientBuilder.build(), bucket)!=null) {
                log.info("S3 endpoint for " + bucket + " set to endpoint of region " + regionFromLocation);
            } else {
                s3ClientBuilder.withRegion(oldRegion);
                log.info("S3 endpoint for " + bucket + " left at endpoint of region " + s3ClientBuilder.getRegion());
            }
        } catch (final IllegalArgumentException e) {
            log.warn("Unable to get bucket location, using default.", e);
        }
        return s3ClientBuilder;
    }

    public static void setBestEndpointForBucket(final AmazonS3 s3Client, final String bucket) {
        log.info("Detecting bucket location for [" + bucket + "] via " + s3Client.getRegion().toAWSRegion());
        final String bucketLocation = getBucketLocation(s3Client, bucket);
        if (bucketLocation==null) {
            return;
        }
        try {
            final Region region = Region.fromValue(bucketLocation);
            final com.amazonaws.regions.Region awsRegion = region.toAWSRegion();
            final Region oldregion = s3Client.getRegion();
            s3Client.setRegion(awsRegion);
            if (getBucketLocation(s3Client, bucket)!=null) {
                log.info("S3 endpoint for " + bucket + " set to: " + S3Support.getEndpoint(awsRegion));
            } else {
                log.info("S3 endpoint for " + bucket + " left at: " + S3Support.getEndpoint(awsRegion));
                s3Client.setRegion(oldregion.toAWSRegion());
            }
        } catch (final IllegalArgumentException e) {
            log.warn("Unable to get bucket location, using default.", e);
        }
    }

    @Nullable
    private static String getBucketLocation(final AmazonS3 s3Client, final String bucketName) {
        try {
            return s3Client.getBucketLocation(bucketName);
        } catch (final AmazonClientException e) {
            log.warn("Unable to get bucket location for [" + bucketName + "], using default. Error: " + e);
            log.debug("", e);
            return null;
        }
    }

    private static Regions toRegions(final com.amazonaws.regions.Region region) {
        return Regions.fromName(region.getName());
    }
}
