/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.audit.broker;

import com.atlassian.audit.api.AuditConsumer;
import com.atlassian.audit.broker.AuditConsumerExceptionHandler;
import com.atlassian.audit.broker.AuditEntityRejectionHandler;
import com.atlassian.audit.broker.AuditPolicy;
import com.atlassian.audit.broker.InternalAuditBroker;
import com.atlassian.audit.denylist.ExcludedActionsService;
import com.atlassian.audit.entity.AuditEntity;
import com.atlassian.audit.event.AuditConsumerAddedEvent;
import com.atlassian.audit.event.AuditConsumerRemovedEvent;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import javax.annotation.Nonnull;
import javax.annotation.concurrent.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

@ThreadSafe
public class ScatterAuditBroker
implements InternalAuditBroker,
InitializingBean,
DisposableBean {
    private static final Logger log = LoggerFactory.getLogger(ScatterAuditBroker.class);
    private static final String IGNORE_AUDIT_POLICY_ANNOTATION_CLASSNAME = "com.atlassian.audit.api.IgnoreAuditPolicy";
    private static final String IGNORE_EXCLUDED_AUDIT_POLICY_ANNOTATION_CLASSNAME = "com.atlassian.audit.api.IgnoreAuditExclusionsPolicy";
    private final EventPublisher eventPublisher;
    private final AuditPolicy auditPolicy;
    private final AuditEntityRejectionHandler rejectAuditEntityHandler;
    private final AuditConsumerExceptionHandler exceptionHandler;
    private final ConcurrentHashMap<AuditConsumer, ConsumerRegistration> consumerRegistry;
    private final int defaultConsumerBufferSize;
    private final int defaultConsumerBatchSize;
    private final ExcludedActionsService excludedActionsService;

    public ScatterAuditBroker(EventPublisher eventPublisher, AuditPolicy auditPolicy, AuditEntityRejectionHandler rejectionHandler, AuditConsumerExceptionHandler exceptionHandler, int defaultConsumerBufferSize, int defaultConsumerBatchSize, ExcludedActionsService excludedActionsService) {
        this.eventPublisher = eventPublisher;
        this.defaultConsumerBatchSize = defaultConsumerBatchSize;
        this.defaultConsumerBufferSize = defaultConsumerBufferSize;
        this.auditPolicy = Objects.requireNonNull(auditPolicy);
        this.rejectAuditEntityHandler = Objects.requireNonNull(rejectionHandler);
        this.exceptionHandler = Objects.requireNonNull(exceptionHandler);
        this.consumerRegistry = new ConcurrentHashMap();
        this.excludedActionsService = excludedActionsService;
    }

    public void afterPropertiesSet() {
        this.eventPublisher.register((Object)this);
    }

    public void destroy() {
        this.shutdown();
    }

    public synchronized void shutdown() {
        this.consumerRegistry.values().forEach(registry -> registry.getThread().shutdown());
        this.waitForTermination();
    }

    public synchronized void shutdownNow() {
        this.consumerRegistry.values().forEach(registry -> registry.getThread().shutdownNow());
        this.waitForTermination();
    }

    @EventListener
    public void onAuditConsumerAdded(AuditConsumerAddedEvent event) {
        this.addConsumer(event.getConsumerService(), this.defaultConsumerBufferSize, this.defaultConsumerBatchSize);
    }

    @EventListener
    public void onAuditConsumerRemoved(AuditConsumerRemovedEvent event) {
        this.removeConsumer(event.getConsumerService(), false);
    }

    public void addConsumer(AuditConsumer consumer, int bufferSize, int batchSize) {
        ConsumerQueue queue = new ConsumerQueue(consumer, new ArrayBlockingQueue<AuditEntity>(bufferSize), batchSize, entity -> this.rejectAuditEntityHandler.reject(this, consumer, (List<AuditEntity>)entity));
        ConsumerThread thread = new ConsumerThread(queue, consumer, (exception, batch) -> this.exceptionHandler.handle(consumer, (RuntimeException)exception, (List<AuditEntity>)batch));
        log.trace("#addConsumer consumer={}, bufferSize={}, batchSize={}, thread={}", new Object[]{consumer, bufferSize, batchSize, thread.getName()});
        this.consumerRegistry.put(consumer, new ConsumerRegistration(queue, thread, this.getPolicyAdherenceBehaviour(consumer)));
        thread.start();
    }

    private PolicyAdherence getPolicyAdherenceBehaviour(AuditConsumer consumer) {
        boolean consumerIgnoresAuditCoveragePolicy = ScatterAuditBroker.consumerContainsAnnotation(consumer, IGNORE_AUDIT_POLICY_ANNOTATION_CLASSNAME);
        boolean consumerIgnoresAuditExclusionPolicy = ScatterAuditBroker.consumerContainsAnnotation(consumer, IGNORE_EXCLUDED_AUDIT_POLICY_ANNOTATION_CLASSNAME);
        return PolicyAdherence.findPolicyAdherence(consumerIgnoresAuditCoveragePolicy, consumerIgnoresAuditExclusionPolicy);
    }

    private static boolean consumerContainsAnnotation(AuditConsumer consumer, String annotationToConsider) {
        return Arrays.stream(consumer.getClass().getAnnotations()).anyMatch(annotation -> annotation.annotationType().getName().equals(annotationToConsider));
    }

    public void removeConsumer(AuditConsumer consumer, boolean force) {
        ConsumerRegistration registration = this.consumerRegistry.remove(consumer);
        log.trace("#removeConsumer consumer={}, force={}, registration={}", new Object[]{consumer, force, registration});
        if (registration != null) {
            if (force) {
                registration.getThread().shutdownNow();
            } else {
                registration.getThread().shutdown();
            }
        }
    }

    @Override
    public void audit(@Nonnull AuditEntity entity) {
        Objects.requireNonNull(entity, "entity");
        boolean passedAuditPolicy = this.auditPolicy.pass(entity);
        boolean passedExclusionPolicy = !this.isEntityExcluded(entity);
        this.consumerRegistry.entrySet().stream().filter(registry -> ((AuditConsumer)registry.getKey()).isEnabled()).filter(registry -> ((ConsumerRegistration)registry.getValue()).getPolicyAdherence().shouldBeConsumed(passedAuditPolicy, passedExclusionPolicy)).forEach(registry -> ((ConsumerRegistration)registry.getValue()).queue.offer(entity));
        this.traceLog("#audit auditPolicy.pass={}, auditExclusionPolicy.pass={}, entity={}", passedAuditPolicy, passedExclusionPolicy, entity);
    }

    private boolean isEntityExcluded(AuditEntity entity) {
        return this.excludedActionsService.shouldExclude(entity);
    }

    private void waitForTermination() {
        this.consumerRegistry.values().forEach(consumerRegistration -> {
            try {
                consumerRegistration.thread.join();
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        });
    }

    private void traceLog(String message, Object ... args) {
        if (log.isTraceEnabled()) {
            log.trace(message, args);
        }
    }

    @ThreadSafe
    private static final class ConsumerQueue {
        private final AuditConsumer auditConsumer;
        private final BlockingQueue<AuditEntity> queue;
        private final int batchSize;
        private final Consumer<List<AuditEntity>> rejectionHandler;

        ConsumerQueue(AuditConsumer auditConsumer, BlockingQueue<AuditEntity> queue, int batchSize, Consumer<List<AuditEntity>> rejectionHandler) {
            this.auditConsumer = auditConsumer;
            this.queue = Objects.requireNonNull(queue);
            this.batchSize = batchSize;
            this.rejectionHandler = Objects.requireNonNull(rejectionHandler);
        }

        void offer(AuditEntity entity) {
            while (!this.queue.offer(entity)) {
                this.discardOldestEntities();
            }
            log.trace("#offer auditConsumer={}, entity={}", (Object)this.auditConsumer, (Object)entity);
        }

        void clear() {
            this.queue.clear();
        }

        List<AuditEntity> take() throws InterruptedException {
            ArrayList<AuditEntity> batch = new ArrayList<AuditEntity>(this.batchSize);
            AuditEntity entity = this.queue.take();
            batch.add(entity);
            while (batch.size() < this.batchSize && (entity = (AuditEntity)this.queue.poll()) != null) {
                batch.add(entity);
            }
            return batch;
        }

        List<AuditEntity> poll() {
            AuditEntity entity;
            ArrayList<AuditEntity> batch = new ArrayList<AuditEntity>(this.batchSize);
            while (batch.size() < this.batchSize && (entity = (AuditEntity)this.queue.poll()) != null) {
                batch.add(entity);
            }
            return batch;
        }

        private void discardOldestEntities() {
            AuditEntity entity;
            ArrayList<AuditEntity> batch = new ArrayList<AuditEntity>(this.batchSize);
            for (int i = 0; i < this.batchSize && (entity = (AuditEntity)this.queue.poll()) != null; ++i) {
                batch.add(entity);
            }
            log.trace("#discardOldestEntities auditConsumer={}, batchSize={}, batch={}", new Object[]{this.auditConsumer, this.batchSize, batch});
            this.rejectionHandler.accept(batch);
        }
    }

    private final class ConsumerThread
    extends Thread {
        private final AtomicBoolean running;
        private final AuditConsumer consumer;
        private final ConsumerQueue queue;
        private final BiConsumer<RuntimeException, List<AuditEntity>> exceptionHandler;

        ConsumerThread(ConsumerQueue queue, AuditConsumer consumer, BiConsumer<RuntimeException, List<AuditEntity>> exceptionHandler) {
            super("audit-broker-consumer-thread-" + Integer.toHexString(ScatterAuditBroker.this.hashCode()) + "-consumer-" + Integer.toHexString(consumer.hashCode()));
            this.running = new AtomicBoolean(false);
            this.queue = Objects.requireNonNull(queue);
            this.consumer = Objects.requireNonNull(consumer);
            this.exceptionHandler = Objects.requireNonNull(exceptionHandler);
        }

        @Override
        public void run() {
            try {
                while (!this.isInterrupted()) {
                    try {
                        List<AuditEntity> batch = this.queue.take();
                        log.trace("#run batch={}", batch);
                        this.processBatch(batch);
                    }
                    catch (InterruptedException e) {
                        // empty catch block
                        break;
                    }
                }
                log.trace("#run ConsumerThread interrupted, consumer={}, queue={}", (Object)this.consumer, this.queue.queue);
            }
            catch (Throwable uncaughtThrowable) {
                log.error("#run ConsumerThread killed by an uncaught throwable", uncaughtThrowable);
                throw uncaughtThrowable;
            }
        }

        @Override
        public synchronized void start() {
            if (this.running.compareAndSet(false, true)) {
                super.start();
            }
        }

        public void shutdown() {
            if (this.running.compareAndSet(true, false)) {
                this.interrupt();
                this.drainQueue();
            }
        }

        public void shutdownNow() {
            if (this.running.compareAndSet(true, false)) {
                this.interrupt();
            }
            this.queue.clear();
            this.running.set(false);
        }

        private void drainQueue() {
            List<AuditEntity> batch;
            while (!(batch = this.queue.poll()).isEmpty()) {
                log.trace("#drainQueue batch={}", batch);
                this.processBatch(batch);
            }
        }

        private void processBatch(List<AuditEntity> batch) {
            try {
                this.consumer.accept(batch);
            }
            catch (RuntimeException e) {
                this.exceptionHandler.accept(e, batch);
            }
        }
    }

    private static class ConsumerRegistration {
        private final ConsumerQueue queue;
        private final ConsumerThread thread;
        private final PolicyAdherence policyAdherence;

        private ConsumerRegistration(ConsumerQueue queue, ConsumerThread thread, PolicyAdherence policyAdherence) {
            this.queue = Objects.requireNonNull(queue);
            this.thread = Objects.requireNonNull(thread);
            this.policyAdherence = policyAdherence;
        }

        ConsumerThread getThread() {
            return this.thread;
        }

        PolicyAdherence getPolicyAdherence() {
            return this.policyAdherence;
        }
    }

    static enum PolicyAdherence {
        ALLOW_ALL(true, true, (passedAuditPolicy, passedExclusionPolicy) -> true),
        DEFAULT(false, false, (passedAuditPolicy, passedExclusionPolicy) -> passedAuditPolicy != false && passedExclusionPolicy != false),
        IGNORE_AUDIT_COVERAGE_POLICY(true, false, (passedAuditPolicy, passedExclusionPolicy) -> passedExclusionPolicy),
        IGNORE_AUDIT_EXCLUSION_POLICY(false, true, (passedAuditPolicy, passedExclusionPolicy) -> passedAuditPolicy);

        private final boolean consumerIgnoresAuditCoveragePolicy;
        private final boolean consumerIgnoresAuditExclusionPolicy;
        private final BiFunction<Boolean, Boolean, Boolean> shouldBeAuditedBiFunction;

        private PolicyAdherence(boolean consumerIgnoresAuditCoveragePolicy, boolean consumerIgnoresAuditExclusionPolicy, BiFunction<Boolean, Boolean, Boolean> shouldBeAuditedBiFunction) {
            this.consumerIgnoresAuditCoveragePolicy = consumerIgnoresAuditCoveragePolicy;
            this.consumerIgnoresAuditExclusionPolicy = consumerIgnoresAuditExclusionPolicy;
            this.shouldBeAuditedBiFunction = shouldBeAuditedBiFunction;
        }

        public boolean shouldBeConsumed(boolean passedAuditPolicy, boolean passedExclusionPolicy) {
            return this.shouldBeAuditedBiFunction.apply(passedAuditPolicy, passedExclusionPolicy);
        }

        public static PolicyAdherence findPolicyAdherence(boolean consumerIgnoresAuditCoveragePolicy, boolean consumerIgnoresAuditExclusionPolicy) {
            return Arrays.stream(PolicyAdherence.values()).filter(policyAdherence -> policyAdherence.consumerIgnoresAuditCoveragePolicy == consumerIgnoresAuditCoveragePolicy && policyAdherence.consumerIgnoresAuditExclusionPolicy == consumerIgnoresAuditExclusionPolicy).findFirst().orElse(DEFAULT);
        }
    }
}

