package com.atlassian.audit.search;

import com.atlassian.activeobjects.external.ActiveObjects;
import com.atlassian.audit.ao.dao.entity.AoAuditEntity;
import com.atlassian.audit.ao.dao.entity.AoAuditEntityAction;
import com.atlassian.audit.coverage.SingleValueCache;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import net.java.ao.Query;

import javax.annotation.Nonnull;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import static java.util.concurrent.TimeUnit.SECONDS;

/**
 * Query actions from Audit DB and store it in a Time To Live cache.
 */
public class ActionsProvider {

    // Limit the number of action to be 10,000
    private static final int ACTION_LIMIT = 10_000;

    private final ActiveObjects ao;
    private final TransactionTemplate transactionTemplate;
    private final SingleValueCache<Set<String>> actionsCache;

    public ActionsProvider(ActiveObjects ao,
                           TransactionTemplate transactionTemplate,
                           int refreshIntervalInSeconds) {
        this.ao = ao;
        this.transactionTemplate = transactionTemplate;
        this.actionsCache = new SingleValueCache<>(
                this::queryDistinctActions,
                refreshIntervalInSeconds, SECONDS);
    }

    /**
     * This method is invoked by {@link SingleValueCache} when cache expired to query all distinct action values.
     * Use full scope search is faster than timestamp scoped query, though the latter has less number of rows to scan.
     *
     * @return all distinct actions
     */
    @Nonnull
    private Set<String> queryDistinctActions() {
        return Arrays.stream(transactionTemplate.execute(() -> ao.find(AoAuditEntityAction.class,
                Query.select(AoAuditEntity.ACTION_COLUMN)
                        .distinct()
                        .limit(ACTION_LIMIT))))
                .map(AoAuditEntity::getAction)
                .collect(Collectors.toSet());
    }

    public List<String> getActions() {
        return actionsCache.get().stream().sorted().collect(Collectors.toList());
    }
}
