package com.atlassian.audit.schedule.db.limit;

import com.atlassian.audit.ao.dao.AuditEntityDao;
import com.atlassian.audit.plugin.configuration.PropertiesProvider;
import com.atlassian.scheduler.SchedulerService;
import com.atlassian.scheduler.SchedulerServiceException;
import com.atlassian.scheduler.config.JobConfig;
import com.atlassian.scheduler.config.JobId;
import com.atlassian.scheduler.config.JobRunnerKey;
import com.atlassian.scheduler.config.Schedule;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Instant;
import java.util.Date;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import static com.atlassian.scheduler.config.RunMode.RUN_ONCE_PER_CLUSTER;
import static java.time.temporal.ChronoUnit.MINUTES;

public class DbLimiterScheduler {
    public static final String AUDIT_DB_LIMIT_ROWS_KEY = "plugin.audit.db.limit.rows";
    public static final int AUDIT_DB_LIMIT_ROWS_DEFAULT = 10_000_000;
    //Number of rows to be deleted
    public static final String AUDIT_DB_LIMIT_BUFFER_ROWS_KEY = "plugin.audit.db.limit.buffer.rows";
    public static final int AUDIT_DB_LIMIT_BUFFER_ROWS_DEFAULT = 1000;

    private static final JobRunnerKey AUDIT_DB_LIMITER_JOB_RUNNER_KEY = JobRunnerKey.of(DbLimiterJobRunner.class.getName());
    private static final JobId AUDIT_DB_LIMITER_JOB_ID = JobId.of(DbLimiterJobRunner.class.getName());
    private static final Logger log = LoggerFactory.getLogger(DbLimiterJobRunner.class);
    private static final int DB_LIMITER_SCHEDULE_INTERVAL_DEFAULT_MINS = 60;
    private static final String DB_LIMITER_SCHEDULE_INTERVAL_KEY = "plugin.audit.schedule.db.limiter.interval.mins";

    private final SchedulerService schedulerService;
    private final AuditEntityDao auditEntityDao;
    private final Supplier<Integer> jobIntervalInMinuteSupplier;
    private final int rowsLimit;
    private final int rowsLimitBuffer;

    public DbLimiterScheduler(SchedulerService schedulerService,
                              AuditEntityDao auditEntityDao,
                              PropertiesProvider propertiesProvider
    ) {
        this.schedulerService = schedulerService;
        this.auditEntityDao = auditEntityDao;
        this.jobIntervalInMinuteSupplier = () -> propertiesProvider.getInteger(DB_LIMITER_SCHEDULE_INTERVAL_KEY,
                DB_LIMITER_SCHEDULE_INTERVAL_DEFAULT_MINS);
        this.rowsLimit = propertiesProvider.getInteger(AUDIT_DB_LIMIT_ROWS_KEY, AUDIT_DB_LIMIT_ROWS_DEFAULT);
        this.rowsLimitBuffer = propertiesProvider.getInteger(AUDIT_DB_LIMIT_BUFFER_ROWS_KEY, AUDIT_DB_LIMIT_BUFFER_ROWS_DEFAULT);
    }

    public void registerJob() {
        schedulerService.registerJobRunner(AUDIT_DB_LIMITER_JOB_RUNNER_KEY,
                new DbLimiterJobRunner(
                        auditEntityDao,
                        rowsLimit,
                        rowsLimitBuffer
                ));
        JobConfig config = JobConfig.forJobRunnerKey(AUDIT_DB_LIMITER_JOB_RUNNER_KEY)
                .withRunMode(RUN_ONCE_PER_CLUSTER)
                .withSchedule(Schedule.forInterval(TimeUnit.MINUTES.toMillis(jobIntervalInMinuteSupplier.get()),
                        Date.from(Instant.now().plus(jobIntervalInMinuteSupplier.get(), MINUTES))));

        try {
            schedulerService.scheduleJob(AUDIT_DB_LIMITER_JOB_ID, config);
            log.info("Jobrunner {} registered", AUDIT_DB_LIMITER_JOB_RUNNER_KEY);
        } catch (SchedulerServiceException e) {
            log.error("Could not schedule auditing DB limiter job", e);
        }
    }

    public void unregisterJob() {
        schedulerService.unregisterJobRunner(AUDIT_DB_LIMITER_JOB_RUNNER_KEY);
        log.info("Jobrunner {} unregistered", AUDIT_DB_LIMITER_JOB_RUNNER_KEY);
    }

}
