package com.atlassian.audit.retention;

import com.atlassian.audit.analytics.RetentionUpdatedEvent;
import com.atlassian.audit.api.AuditRetentionConfig;
import com.atlassian.audit.api.AuditRetentionConfigService;
import com.atlassian.audit.api.AuditService;
import com.atlassian.audit.api.events.AuditRetentionConfigUpdatedEvent;
import com.atlassian.audit.entity.AuditEvent;
import com.atlassian.audit.entity.ChangedValue;
import com.atlassian.audit.plugin.AuditPluginInfo;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.sal.api.pluginsettings.PluginSettings;
import com.atlassian.sal.api.pluginsettings.PluginSettingsFactory;

import javax.annotation.Nonnull;
import java.time.Period;
import java.util.Optional;

import static com.atlassian.audit.api.AuditRetentionConfig.DEFAULT_RETENTION_PERIOD;
import static com.atlassian.audit.coverage.AuditedCoverageConfigService.AUDIT_CONFIG_UPDATED;
import static java.lang.String.format;
import static java.time.temporal.ChronoUnit.DAYS;
import static java.time.temporal.ChronoUnit.MONTHS;
import static java.time.temporal.ChronoUnit.YEARS;
import static java.util.Objects.requireNonNull;

public class SalAuditRetentionConfigService implements AuditRetentionConfigService {

    private static final String PLUGIN_KEY_RETENTION_PERIOD = "com.atlassian.audit.plugin:audit-config:retention:period";

    private final EventPublisher eventPublisher;
    private final PluginSettingsFactory pluginSettingsFactory;
    private final AuditPluginInfo auditPluginInfo;
    private final AuditService auditService;

    public SalAuditRetentionConfigService(EventPublisher eventPublisher, AuditPluginInfo auditPluginInfo,
                                          PluginSettingsFactory pluginSettingsFactory, AuditService auditService) {
        this.eventPublisher = eventPublisher;
        this.auditPluginInfo = auditPluginInfo;
        this.pluginSettingsFactory = pluginSettingsFactory;
        this.auditService = auditService;
    }

    @Nonnull
    @Override
    public AuditRetentionConfig getConfig() {
        final Period period = Optional.ofNullable((String) getPluginSettings().get(PLUGIN_KEY_RETENTION_PERIOD))
                .map(Period::parse)
                .orElse(DEFAULT_RETENTION_PERIOD);

        return new AuditRetentionConfig(period);
    }

    @Override
    public void updateConfig(@Nonnull AuditRetentionConfig auditRetentionConfig) {
        requireNonNull(auditRetentionConfig, "auditRetentionConfig");

        PluginSettings pluginSettings = getPluginSettings();
        String oldRetentionConfig = (String) pluginSettings.get(PLUGIN_KEY_RETENTION_PERIOD);
        pluginSettings.put(PLUGIN_KEY_RETENTION_PERIOD, auditRetentionConfig.getPeriod().toString());

        eventPublisher.publish(new AuditRetentionConfigUpdatedEvent());
        if (!auditRetentionConfig.getPeriod().toString().equals(oldRetentionConfig)) {
            Period oldPeriod = oldRetentionConfig == null ? null : Period.parse(oldRetentionConfig);
            Period newPeriod = auditRetentionConfig.getPeriod();
            auditService.audit(AuditEvent.builder(AUDIT_CONFIG_UPDATED)
                    .changedValue(ChangedValue.fromI18nKeys("atlassian.audit.event.change.retention")
                                    .from(oldPeriod == null ? null : format("%s %s", getPeriodValue(oldPeriod), getPeriodUnit(oldPeriod)))
                                    .to(format("%s %s", getPeriodValue(newPeriod), getPeriodUnit(newPeriod))).build())
                    .build());
            eventPublisher.publish(new RetentionUpdatedEvent(
                    getPeriodValue(oldPeriod), getPeriodUnit(oldPeriod),
                    getPeriodValue(newPeriod), getPeriodUnit(newPeriod),
                    auditPluginInfo.getPluginVersion()
            ));
        }
    }

    private PluginSettings getPluginSettings() {
        return pluginSettingsFactory.createGlobalSettings();
    }

    private String getPeriodUnit(Period oldPeriod) {
        return oldPeriod == null ? "" : oldPeriod.getYears() > 0 ? YEARS.toString() : oldPeriod.getMonths() > 0 ? MONTHS.toString() : DAYS.toString();
    }

    private String getPeriodValue(Period oldPeriod) {
        return String.valueOf(oldPeriod == null ? "" : oldPeriod.getYears() > 0 ? oldPeriod.getYears() : oldPeriod.getMonths() > 0 ? oldPeriod.getMonths() : oldPeriod.getDays());
    }
}
