package com.atlassian.audit.retention;

import com.atlassian.audit.ao.dao.AuditEntityDao;
import com.atlassian.audit.api.AuditRetentionConfigService;
import com.atlassian.audit.plugin.configuration.PropertiesProvider;
import com.atlassian.sal.api.pluginsettings.PluginSettingsFactory;
import com.atlassian.scheduler.SchedulerService;
import com.atlassian.scheduler.SchedulerServiceException;
import com.atlassian.scheduler.config.JobConfig;
import com.atlassian.scheduler.config.JobId;
import com.atlassian.scheduler.config.JobRunnerKey;
import com.atlassian.scheduler.config.Schedule;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.atlassian.audit.retention.RetentionJobRunner.CLEAN_UP_INTERVAL_IN_HOURS_DEFAULT;
import static com.atlassian.audit.retention.RetentionJobRunner.CLEAN_UP_INTERVAL_IN_HOURS_KEY;
import static com.atlassian.scheduler.config.RunMode.RUN_ONCE_PER_CLUSTER;

public class RetentionScheduler {

    private static final JobRunnerKey AUDIT_CLEANUP_JOB_RUNNER_KEY = JobRunnerKey.of(RetentionJobRunner.class.getName());
    private static final JobId AUDIT_CLEANUP_JOB_ID = JobId.of(RetentionJobRunner.class.getName());
    private static final Logger log = LoggerFactory.getLogger(RetentionJobRunner.class);

    private final SchedulerService schedulerService;
    private final AuditEntityDao auditEntityDao;
    private final AuditRetentionConfigService auditRetentionConfigService;
    private final PluginSettingsFactory settingsFactory;
    private final int cleanUpInterval;

    public RetentionScheduler(SchedulerService schedulerService,
                              AuditRetentionConfigService auditRetentionConfigService,
                              AuditEntityDao auditEntityDao,
                              PluginSettingsFactory settingsFactory,
                              PropertiesProvider propertiesProvider
    ) {
        this.schedulerService = schedulerService;
        this.auditRetentionConfigService = auditRetentionConfigService;
        this.auditEntityDao = auditEntityDao;
        this.settingsFactory = settingsFactory;
        this.cleanUpInterval = propertiesProvider.getInteger(CLEAN_UP_INTERVAL_IN_HOURS_KEY,
                CLEAN_UP_INTERVAL_IN_HOURS_DEFAULT);
    }

    public void registerJob() {
        schedulerService.registerJobRunner(AUDIT_CLEANUP_JOB_RUNNER_KEY,
                new RetentionJobRunner(auditEntityDao, auditRetentionConfigService, settingsFactory, cleanUpInterval));

        JobConfig config = JobConfig.forJobRunnerKey(AUDIT_CLEANUP_JOB_RUNNER_KEY)
                .withRunMode(RUN_ONCE_PER_CLUSTER)
                .withSchedule(Schedule.forCronExpression("0 0 0 1/1 * ? *"));

        try {
            schedulerService.scheduleJob(AUDIT_CLEANUP_JOB_ID, config);
            log.info("Jobrunner {} registered", AUDIT_CLEANUP_JOB_RUNNER_KEY);
        } catch (SchedulerServiceException e) {
            log.error("Could not schedule auditing cleanup job", e);
        }
    }

    public void unregisterJob() {
        schedulerService.unregisterJobRunner(AUDIT_CLEANUP_JOB_RUNNER_KEY);
        log.info("Jobrunner {} unregistered", AUDIT_CLEANUP_JOB_RUNNER_KEY);
    }

}
