package com.atlassian.audit.rest;

import com.atlassian.audit.api.AuditEntityCursor;
import com.atlassian.audit.api.AuditQuery;
import com.atlassian.audit.api.AuditQuery.AuditResourceIdentifier;
import com.atlassian.audit.api.AuditRetentionConfigService;
import com.atlassian.audit.file.AuditRetentionFileConfig;
import com.atlassian.audit.file.AuditRetentionFileConfigService;
import com.atlassian.audit.api.AuditSearchService;
import com.atlassian.audit.api.util.pagination.Page;
import com.atlassian.audit.api.util.pagination.PageRequest;
import com.atlassian.audit.coverage.InternalAuditCoverageConfigService;
import com.atlassian.audit.coverage.ProductLicenseChecker;
import com.atlassian.audit.csv.AuditCsvExportService;
import com.atlassian.audit.csv.AuditCsvExporter;
import com.atlassian.audit.entity.AuditEntity;
import com.atlassian.audit.plugin.configuration.PermissionsEnforced;
import com.atlassian.audit.plugin.configuration.PropertiesProvider;
import com.atlassian.audit.rest.model.AuditCoverageConfigJson;
import com.atlassian.audit.rest.model.AuditEntitiesResponseJson;
import com.atlassian.audit.rest.model.AuditRetentionFileConfigJson;
import com.atlassian.audit.rest.model.AuditRetentionConfigJson;
import com.atlassian.audit.rest.model.ResponseErrorJson;
import com.atlassian.audit.rest.utils.AuditEntitySerializer;
import com.atlassian.audit.rest.validation.AuditRestValidator.ActionsValidator;
import com.atlassian.audit.rest.validation.AuditRestValidator.AffectedObjectsValidator;
import com.atlassian.audit.rest.validation.AuditRestValidator.CategoriesValidator;
import com.atlassian.audit.rest.validation.AuditRestValidator.CursorValidator;
import com.atlassian.audit.rest.validation.AuditRestValidator.FormatValidator;
import com.atlassian.audit.rest.validation.AuditRestValidator.FromValidator;
import com.atlassian.audit.rest.validation.AuditRestValidator.LimitValidator;
import com.atlassian.audit.rest.validation.AuditRestValidator.OffsetValidator;
import com.atlassian.audit.rest.validation.AuditRestValidator.ScanLimitValidator;
import com.atlassian.audit.rest.validation.AuditRestValidator.SearchValidator;
import com.atlassian.audit.rest.validation.AuditRestValidator.ToValidator;
import com.atlassian.audit.rest.validation.AuditRestValidator.UserIdsValidator;
import com.atlassian.audit.rest.validation.ValidationInterceptor;
import com.atlassian.audit.rest.validation.Validator;
import com.atlassian.audit.spi.entity.AuditEntityTransformationService;
import com.atlassian.plugins.rest.common.interceptor.InterceptorChain;
import com.atlassian.sal.api.ApplicationProperties;
import com.atlassian.sal.api.timezone.TimeZoneManager;
import com.sun.jersey.core.header.ContentDisposition;
import io.swagger.v3.oas.annotations.OpenAPIDefinition;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.info.Info;
import io.swagger.v3.oas.annotations.media.ArraySchema;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.ws.rs.Consumes;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.core.StreamingOutput;
import javax.ws.rs.core.UriInfo;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeoutException;
import java.util.stream.Stream;

import static com.atlassian.audit.retention.SalAuditRetentionFileConfigService.DEFAULT_MAX_FILE_SIZE_IN_MB;
import static com.atlassian.sal.api.UrlMode.CANONICAL;
import static io.swagger.v3.oas.annotations.enums.Explode.FALSE;
import static io.swagger.v3.oas.annotations.enums.ParameterStyle.FORM;
import static java.lang.Long.parseLong;
import static java.time.Instant.ofEpochMilli;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON;

@OpenAPIDefinition(info = @Info(title = "Audit", version = "0.0.1", description = "This is a draft of the proposed " +
        "cross-product APIs, which will be supported by Bitbucket, Confluence and Jira. The root path is /rest/auditing/1.0"))
@Path("/")
@Produces(APPLICATION_JSON)
@Consumes(APPLICATION_JSON)
public class AuditRestResource {

    public static final String FORMAT_CSV_FILE = "csv";
    public static final String FORMAT_JSON = "json";

    private static final String TEXT_CSV = "text/csv";
    private static final String CONTENT_DISPOSITION = "Content-Disposition";

    private final AuditSearchService searchService;
    private final InternalAuditCoverageConfigService coverageConfigService;
    private final AuditRetentionConfigService retentionConfigService;
    private final AuditRetentionFileConfigService retentionFileConfigService;
    private final AuditEntityTransformationService transformationService;
    private final AuditCsvExportService auditCsvExportService;
    private final TimeZoneManager timeZoneManager;
    private final ApplicationProperties applicationProperties;
    private final ProductLicenseChecker licenseChecker;
    private final PropertiesProvider propertiesProvider;

    public AuditRestResource(AuditSearchService searchService,
                             @PermissionsEnforced InternalAuditCoverageConfigService coverageConfigService,
                             @PermissionsEnforced AuditRetentionConfigService retentionConfigService,
                             @PermissionsEnforced AuditRetentionFileConfigService retentionFileConfigService,
                             AuditEntityTransformationService transformationService,
                             AuditCsvExportService auditCsvExportService,
                             TimeZoneManager timeZoneManager,
                             ApplicationProperties applicationPropertiesSupplier,
                             ProductLicenseChecker licenseChecker,
                             PropertiesProvider propertiesProvider) {
        this.searchService = searchService;
        this.coverageConfigService = coverageConfigService;
        this.retentionConfigService = retentionConfigService;
        this.retentionFileConfigService = retentionFileConfigService;
        this.auditCsvExportService = auditCsvExportService;
        this.transformationService = transformationService;
        this.timeZoneManager = timeZoneManager;
        this.applicationProperties = applicationPropertiesSupplier;
        this.licenseChecker = licenseChecker;
        this.propertiesProvider = propertiesProvider;
    }

    @GET
    @Path("/events")
    @Produces({APPLICATION_JSON, TEXT_CSV})
    @Operation(summary = "Get a paginated list of audit events", tags = {"audit"})
    @InterceptorChain({ValidationInterceptor.class})
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Successful operation", content = @Content(schema =
            @Schema(implementation = AuditEntitiesResponseJson.class))),
            @ApiResponse(responseCode = "400", description = "Bad request", content = @Content(array =
            @ArraySchema(schema = @Schema(implementation = ResponseErrorJson.class))))})
    public Response getAuditEvents(
            @Parameter(description = "The start timestamp in ISO8601 format", example = "2019-11-01T01:00:00.000Z")
            @Schema(type = "string", format = "date-time")
            @QueryParam("from")
            @Validator(FromValidator.class)
                    String from,
            @Parameter(description = "The end timestamp in ISO8601 format", example = "2019-12-01T01:00:00.000Z")
            @Schema(type = "string", format = "date-time")
            @QueryParam("to")
            @Validator(ToValidator.class)
                    String to,
            @Parameter(description = "The number of records to skip")
            @Schema(minimum = "0", type = "integer", format = "int32")
            @QueryParam("offset")
            @DefaultValue("0")
            @Validator(OffsetValidator.class)
                    String rawOffset,
            @Parameter(description = "Location of last result returned in format of timestamp,ID. For making a request for page X, " +
                    "the value of this field can be obtained from pagingInfo->nextPageCursor in response for page X-1",
                    example = "1577437517322,9")
            @QueryParam("pageCursor")
            @Validator(CursorValidator.class)
                    String cursor,
            @Parameter(description = "The maximum number of records returned")
            @Schema(minimum = "1", maximum = "100000", type = "integer", format = "int32")
            @QueryParam("limit")
            @DefaultValue("200")
            @Validator(LimitValidator.class)
                    String rawLimit,
            @Parameter(description = "Audit event author identifiers separated by comma",
                    example = "42,46", style = FORM, explode = FALSE)
            @QueryParam("userIds")
            @Validator(UserIdsValidator.class)
                    String userIds,
            @Parameter(description = "Audit categories separated by comma",
                    example = "Global settings changed,Group deleted", style = FORM, explode = FALSE)
            @QueryParam("categories")
            @Validator(CategoriesValidator.class)
                    String categories,
            @Parameter(description = "Comma-separated list of actions which triggered the audit record",
                    example = "Permissions,Apps", style = FORM, explode = FALSE)
            @QueryParam("actions")
            @Validator(ActionsValidator.class)
                    String actions,
            @Parameter(description = "A list of affected objects separated by semicolon. Each affected object is a pair " +
                    "of object type and id separated by comma. Administrator permission of all affected objects is required when specified. " +
                    "Global administrator permission is required when no affected object is specified.", example = "space,42;space,46")
            @QueryParam("affectedObject")
            @Validator(AffectedObjectsValidator.class)
                    String affectedObject,
            @Parameter(description = "Search expression, this parameter may have negative performance impact. It's recommended " +
                    "to use scanLimit when this parameter is specified.")
            @QueryParam("search")
            @Validator(SearchValidator.class)
                    String search,
            @Parameter(description = "What format output should the server create")
            @QueryParam("outputFormat")
            @DefaultValue(FORMAT_JSON)
            @Validator(FormatValidator.class)
                    String outputFormat,
            @Parameter(description = "The maximum number of records to be scanned in the inverse insertion order " +
                    "with from and to filters taking precedence, " +
                    "the default value is 2147483647 which means there is no limit")
            @QueryParam("scanLimit")
            @Schema(minimum = "1", maximum = "2147483647", type = "integer", format = "int32")
            @DefaultValue("2147483647")
            @Validator(ScanLimitValidator.class)
                    String rawScanLimit,
            @Context HttpHeaders headers,
            @Context SecurityContext securityContext,
            @Context UriInfo uriInfo) {
        int offset = Integer.parseInt(rawOffset);
        int limit = Integer.parseInt(rawLimit);
        int scanLimit = Integer.parseInt(rawScanLimit);

        AuditQuery query = generateAuditQuery(from, to, userIds, categories, actions, affectedObject, search);
        PageRequest<AuditEntityCursor> pageRequest = generatePageRequest(cursor, offset, limit);

        checkDcOnlyFilters(query);

        if (outputFormat.equals(FORMAT_CSV_FILE)) {
            return generateCsvResponse(offset, limit, query);

        } else if (outputFormat.equals((FORMAT_JSON))) {
            return generateJsonResponse(applicationProperties.getBaseUrl(CANONICAL), uriInfo, pageRequest, query, scanLimit);
        }
        throw new IllegalStateException("Unexpected outputFormat was provided and was not caught by validation");
    }

    private void checkDcOnlyFilters(AuditQuery query) {
        //check license for DC only filters
        if ((!query.getCategories().isEmpty() || !query.getActions().isEmpty()) && licenseChecker.isNotDcLicense()) {
            throw new IllegalArgumentException("categories and actions are only supported for Datacenter license.");
        }
    }

    @Nonnull
    private PageRequest<AuditEntityCursor> generatePageRequest(String cursor, int offset, int limit) {
        AuditEntityCursor entityCursor = null;

        if (cursor != null && !cursor.trim().isEmpty()) {
            String[] cursorParts = cursor.split(",\\s*");
            if (cursorParts.length == 2) {
                entityCursor = new AuditEntityCursor(ofEpochMilli(parseLong(cursorParts[0])), parseLong(cursorParts[1]));
            }
        }

        return new PageRequest.Builder<AuditEntityCursor>()
                .offset(offset)
                .limit(limit)
                .cursor(entityCursor)
                .build();
    }

    private AuditQuery generateAuditQuery(String from, String to, String userIds, String categories, String actions, String affectedObjects, String search) {
        final List<AuditResourceIdentifier> resourceIdentifiers = new ArrayList<>();
        if (affectedObjects != null) {
            Stream.of(affectedObjects.split(";\\s*")).forEach(objectString -> {
                final String[] affectedObjectParts = objectString.split(",\\s*");
                // AuditRestValidator.validateAffectedObjects() has validated affectedObjectParts
                resourceIdentifiers.add(new AuditResourceIdentifier(affectedObjectParts[0], affectedObjectParts[1]));
            });
        }

        return AuditQuery.builder()
                .actions(split(actions))
                .userIds(split(userIds))
                .categories(split(categories))
                .searchText(search)
                .from(parseTime(from))
                .to(parseTime(to))
                .resources(resourceIdentifiers)
                .build();
    }

    private Response generateJsonResponse(String baseUrl, @Context UriInfo uriInfo, PageRequest<AuditEntityCursor> pageRequest, AuditQuery query, int scanLimit) {
        try {
            Page<AuditEntity, AuditEntityCursor> entitiesPage = retrieveQuery(pageRequest, query, scanLimit);
            final AuditEntitiesResponseJson response = new AuditEntitiesResponseJson(entitiesPage,
                    entity -> AuditEntitySerializer.toJson(entity, timeZoneManager.getDefaultTimeZone()),
                    baseUrl,
                    uriInfo);
            return Response.ok(response)
                    .type(APPLICATION_JSON)
                    .build();
        } catch (TimeoutException e) {
            return Response.serverError().entity(e.getMessage())
                    .build();
        }
    }

    private Response generateCsvResponse(int offset, int limit, AuditQuery query) {
        final AuditCsvExporter csvExporter;

        csvExporter = auditCsvExportService.createExporter(query);

        //We use a modified ISO8601 format which omits microseconds
        //We use _ since : is an invalid file name character for windows
        String date = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH_mm_ss'Z'").format(LocalDateTime.now(ZoneId.systemDefault()));
        String fileName = String.format("Auditing Export %s.csv", date);

        ContentDisposition contentDisposition = ContentDisposition
                .type("attachment")
                .fileName(fileName)
                .creationDate(new Date(Instant.now().toEpochMilli()))
                .build();

        return Response.ok((StreamingOutput) output -> csvExporter.export(output, offset, limit))
                .type(TEXT_CSV)
                .header(CONTENT_DISPOSITION, contentDisposition)
                .build();
    }

    private Page<AuditEntity, AuditEntityCursor> retrieveQuery(PageRequest<AuditEntityCursor> pageRequest, AuditQuery query, int scanLimit) throws TimeoutException {
        Page<AuditEntity, AuditEntityCursor> page = searchService.findBy(query, pageRequest, scanLimit);

        // Allow products to transform, aka massage things like changing URIs back into the Entity before displaying to users.
        return new Page.Builder<AuditEntity, AuditEntityCursor>(transformationService.transform(page.getValues()), page.getIsLastPage())
                .nextPageRequest(page.getNextPageRequest().orElse(null)).build();
    }

    @Nullable
    private String[] split(String str) {
        if (str == null || str.trim().isEmpty()) {
            return null;
        }
        return str.split(",\\s*");
    }

    @Nullable
    private Instant parseTime(String str) {
        if (str == null) {
            return null;
        }
        return Instant.parse(str);
    }

    @GET
    @Path("/configuration/retention")
    @Operation(summary = "Get current audit log retention database configuration", tags = {"configuration"})
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Successful operation", content = @Content(schema =
            @Schema(implementation = AuditRetentionConfigJson.class))),
            @ApiResponse(responseCode = "400", description = "Bad request", content = @Content(array = @ArraySchema
                    (schema = @Schema(implementation = ResponseErrorJson.class))))})
    public Response getAuditRetentionConfiguration(@Context SecurityContext securityContext) {
        AuditRetentionConfigJson response = new AuditRetentionConfigJson(retentionConfigService.getConfig());
        return Response.ok(response).build();
    }

    @PUT
    @Path("/configuration/retention")
    @Operation(summary = "Set current audit log retention database configuration", tags = {"configuration"})
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Successful operation", content = @Content(schema =
            @Schema(implementation = AuditRetentionConfigJson.class))),
            @ApiResponse(responseCode = "400", description = "Bad request", content = @Content(array = @ArraySchema
                    (schema = @Schema(implementation = ResponseErrorJson.class))))})
    public Response updateAuditRetentionConfiguration(
            @Parameter(required = true) AuditRetentionConfigJson body,
            @Context SecurityContext securityContext) {
        retentionConfigService.updateConfig(body.toRetentionConfig());
        return getAuditRetentionConfiguration(securityContext);
    }

    @GET
    @Path("/configuration/retention/file")
    @Operation(summary = "Get current audit log retention file configuration", tags = {"configuration"})
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Successful operation", content = @Content(schema =
            @Schema(implementation = AuditRetentionFileConfigJson.class))),
            @ApiResponse(responseCode = "400", description = "Bad request", content = @Content(array = @ArraySchema
                    (schema = @Schema(implementation = ResponseErrorJson.class))))})
    public Response getAuditRetentionFileConfiguration(@Context SecurityContext securityContext) {
        AuditRetentionFileConfigJson response = retentionFileConfigService.getConfig().toJson();
        return Response.ok(response).build();
    }

    @PUT
    @Path("/configuration/retention/file")
    @Operation(summary = "Set current audit log retention file configuration", tags = {"configuration"})
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Successful operation", content = @Content(schema =
            @Schema(implementation = AuditRetentionFileConfigJson.class))),
            @ApiResponse(responseCode = "400", description = "Bad request", content = @Content(array = @ArraySchema
                    (schema = @Schema(implementation = ResponseErrorJson.class))))})
    public Response updateAuditRetentionFileConfiguration(
            @Parameter(required = true) AuditRetentionFileConfigJson body,
            @Context SecurityContext securityContext) {
        retentionFileConfigService.updateConfig(AuditRetentionFileConfig.fromJson(body,
                propertiesProvider.getInteger("plugin.audit.file.max.file.size", DEFAULT_MAX_FILE_SIZE_IN_MB)));
        return getAuditRetentionFileConfiguration(securityContext);
    }

    @GET
    @Path("/configuration/coverage")
    @Operation(summary = "Get current audit log coverage configuration", tags = {"configuration"})
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Successful operation", content = @Content(schema =
            @Schema(implementation = AuditCoverageConfigJson.class))),
            @ApiResponse(responseCode = "400", description = "Bad request", content = @Content(array = @ArraySchema
                    (schema = @Schema(implementation = ResponseErrorJson.class))))})
    public Response getAuditCoverageConfiguration(@Context SecurityContext securityContext) {
        AuditCoverageConfigJson response = new AuditCoverageConfigJson(coverageConfigService.getConfig());
        return Response.ok(response).build();
    }

    @PUT
    @Path("/configuration/coverage")
    @Operation(summary = "Set current audit log coverage configuration", tags = {"configuration"})
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Successful operation", content = @Content(schema =
            @Schema(implementation = AuditCoverageConfigJson.class))),
            @ApiResponse(responseCode = "400", description = "Bad request", content = @Content(array = @ArraySchema
                    (schema = @Schema(implementation = ResponseErrorJson.class))))})
    public Response updateAuditCoverageConfiguration(
            @Parameter(required = true) AuditCoverageConfigJson body,
            @Context SecurityContext securityContext) {
        coverageConfigService.updateConfig(body.toCoverageConfig());
        return getAuditCoverageConfiguration(securityContext);
    }
}
