package com.atlassian.audit.plugin.onboarding;

import com.atlassian.audit.permission.PermissionChecker;
import com.atlassian.sal.api.pluginsettings.PluginSettings;
import com.atlassian.sal.api.pluginsettings.PluginSettingsFactory;
import com.atlassian.sal.api.user.UserManager;

/**
 * This is an onboarding notification service.
 */
public class OnboardingSeenService {

    //Caution: Never ever rename or change the value.
    private static final String ONBOARDING_SEEN_PREFIX = "com.atlassian.audit.plugin:audit-config:onboarding:%d:isseen";

    private static final String ONBOARDING_DARKFEATURE_FLAG = "atlassian.darkfeature.audit.onboarding.feature.disabled";

    private final UserManager userManager;

    private final PluginSettingsFactory pluginSettingsFactory;
    
    private final PermissionChecker permissionChecker;
    
    public OnboardingSeenService(final UserManager userManager,
                                 final PluginSettingsFactory pluginSettingsFactory,
                                 final PermissionChecker permissionChecker) {
        this.userManager = userManager;
        this.pluginSettingsFactory = pluginSettingsFactory;
        this.permissionChecker = permissionChecker;
    }

    /**
     * case1: Don't show, if user is not an admin.
     * case2: Don't show if the onboarding has already been see.
     * case3: Show, if it is fresh install (or somehow the configuration has bug or gone then the onboarding will be
     * shown again, which we don't want it to happen)
     *
     * @return whether the onboarding should be shown or not
     * return FALSE if the user is not admin
     * OR the user saw it before in the past.
     * return TRUE if it is fresh install
     * OR the user has never saw it before
     */
    public boolean shouldDisplay() {
        return !isOnboardingDisable() && isOnboardingTargetUser() && !getSeenValue();
    }

    /**
     * Set the GlobalSetting of the onboarding with key = @see com.atlassian.audit.plugin.onboarding.
     * OnboardingSeenService#getKey() to true.
     */
    public void seenAndDismissed() {
        if (isOnboardingTargetUser()) {
            updateSeenValue(true);
        }
    }

    /**
     * If the user has unrestricted view permissions then they should also been shown the onboarding benefits modal.
     */
    private boolean isOnboardingTargetUser() {
        return permissionChecker.hasUnrestrictedAuditViewPermission();
    }

    /**
     * Generate the key
     * Caution: Never ever change the logic of the method
     * @return com.atlassian.audit.plugin:audit-config:onboarding:{usernameHashCode}:isseen
     */
    private String getKey() {
        final int usernameCache = userManager.getRemoteUserKey().hashCode();
        return String.format(ONBOARDING_SEEN_PREFIX, usernameCache);
    }

    private boolean getSeenValue() {
        final PluginSettings settings = pluginSettingsFactory.createGlobalSettings();
        final String isSeen = (String) settings.get(getKey());
        if (isSeen != null) {
            return Boolean.parseBoolean(isSeen);
        }
        return false;
    }

    private void updateSeenValue(boolean isSeen) {
        final PluginSettings settings = pluginSettingsFactory.createGlobalSettings();
        settings.put(getKey(), String.valueOf(isSeen));
    }

    private boolean isOnboardingDisable(){
        final String disableFlag = System.getProperty(ONBOARDING_DARKFEATURE_FLAG);
        return Boolean.parseBoolean(disableFlag);
    }
}
