package com.atlassian.audit.plugin.configuration;

import com.atlassian.activeobjects.external.ActiveObjects;
import com.atlassian.audit.ao.consumer.DatabaseAuditConsumer;
import com.atlassian.audit.ao.dao.AffectedObjectsSerializer;
import com.atlassian.audit.ao.dao.AoAuditEntityDao;
import com.atlassian.audit.ao.dao.AoAuditEntityMapper;
import com.atlassian.audit.ao.dao.AttributesSerializer;
import com.atlassian.audit.ao.dao.AuditEntityDao;
import com.atlassian.audit.ao.dao.AuditEntityMapper;
import com.atlassian.audit.ao.dao.AuditQueryMapper;
import com.atlassian.audit.ao.dao.ChangedValuesSerializer;
import com.atlassian.audit.ao.dao.JacksonAffectedObjectsSerializer;
import com.atlassian.audit.ao.dao.JacksonAttributesSerializer;
import com.atlassian.audit.ao.dao.JacksonChangedValuesSerializer;
import com.atlassian.audit.api.AuditConsumer;
import com.atlassian.audit.file.AuditRetentionFileConfigService;
import com.atlassian.audit.file.CachingRetentionFileConfigService;
import com.atlassian.audit.file.FileAuditConsumer;
import com.atlassian.audit.file.FileMessagePublisher;
import com.atlassian.audit.file.RotatingFileManager;
import com.atlassian.audit.spi.feature.DatabaseAuditingFeature;
import com.atlassian.audit.spi.feature.FileAuditingFeature;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.sal.api.ApplicationProperties;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import org.codehaus.jackson.map.ObjectMapper;
import org.osgi.framework.ServiceRegistration;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import static com.atlassian.audit.file.FileAuditConsumer.DEFAULT_AUDIT_FILE_DIR;
import static com.atlassian.plugins.osgi.javaconfig.ExportOptions.as;
import static com.atlassian.plugins.osgi.javaconfig.OsgiServices.exportOsgiService;
import static com.atlassian.plugins.osgi.javaconfig.OsgiServices.importOsgiService;

@Configuration
public class AuditConsumersConfiguration {
    @SuppressWarnings("rawtypes")
    @Bean
    public FactoryBean<ServiceRegistration> exportDbAuditConsumer(@Qualifier("consumerType-db") AuditConsumer auditConsumer) {
        return exportOsgiService(auditConsumer, as(AuditConsumer.class).withProperty("consumerType", "db"));
    }

    @SuppressWarnings("rawtypes")
    @Bean
    public FactoryBean<ServiceRegistration> exportFileAuditConsumer(@Qualifier("consumerType-file") AuditConsumer auditConsumer) {
        return exportOsgiService(auditConsumer, as(AuditConsumer.class).withProperty("consumerType", "file"));
    }

    @Bean
    public DatabaseAuditingFeature databaseAuditingFeature() {
        return importOsgiService(DatabaseAuditingFeature.class);
    }

    @Bean("consumerType-db")
    public DatabaseAuditConsumer dbConsumer(DatabaseAuditingFeature databaseAuditingFeature,
                                            AuditEntityDao auditEntityDao) {
        return new DatabaseAuditConsumer(databaseAuditingFeature, auditEntityDao);
    }

    @Bean
    public RotatingFileManager getFilePathSupplier(ApplicationProperties appProperties,
                                                   CachingRetentionFileConfigService cachingRetentionFileConfigService) {
        return new RotatingFileManager(appProperties, DEFAULT_AUDIT_FILE_DIR, cachingRetentionFileConfigService);
    }

    @Bean
    public FileMessagePublisher rotatingFileHandler(RotatingFileManager filePathSupplier) {
        return new FileMessagePublisher(filePathSupplier);
    }

    @Bean
    public CachingRetentionFileConfigService cachingRetentionFileConfigService(
            EventPublisher eventPublisher,
            @PermissionsNotEnforced AuditRetentionFileConfigService retentionFileConfigService,
            PropertiesProvider propertiesProvider
    ) {
        final int coverageCacheExpiration = propertiesProvider
                .getInteger("plugin.audit.retention.file.configuration.expiration.seconds", 300);

        return new CachingRetentionFileConfigService(eventPublisher, retentionFileConfigService, coverageCacheExpiration);
    }

    @Bean
    public FileAuditingFeature fileAuditingFeature() {
        return importOsgiService(FileAuditingFeature.class);
    }

    @Bean("consumerType-file")
    public FileAuditConsumer fileConsumer(FileAuditingFeature fileAuditingFeature,
                                          FileMessagePublisher fileMessagePublisher) {
        return new FileAuditConsumer(fileAuditingFeature, fileMessagePublisher);
    }

    @Bean
    public ChangedValuesSerializer changedValuesSerializer(ObjectMapper objectMapper) {
        return new JacksonChangedValuesSerializer(objectMapper);
    }

    @Bean
    public AttributesSerializer attributesSerializer(ObjectMapper objectMapper) {
        return new JacksonAttributesSerializer(objectMapper);
    }

    @Bean
    public AffectedObjectsSerializer affectedObjectsSerializer(ObjectMapper objectMapper) {
        return new JacksonAffectedObjectsSerializer(objectMapper);
    }

    @Bean
    public AuditEntityMapper auditEntityMapper(ChangedValuesSerializer changedValuesSerializer,
                                               AttributesSerializer attributesSerializer,
                                               AffectedObjectsSerializer affectedObjectsSerializer) {
        return new AuditEntityMapper(changedValuesSerializer, attributesSerializer, affectedObjectsSerializer);
    }

    @Bean
    public AoAuditEntityMapper aoAuditEntityMapper(ChangedValuesSerializer changedValuesSerializer,
                                                   AttributesSerializer attributesSerializer,
                                                   AffectedObjectsSerializer affectedObjectsSerializer) {
        return new AoAuditEntityMapper(changedValuesSerializer, attributesSerializer, affectedObjectsSerializer);
    }

    @Bean
    public ActiveObjects ao() {
        return importOsgiService(ActiveObjects.class);
    }

    @Bean
    public TransactionTemplate transactionTemplate() {
        return importOsgiService(TransactionTemplate.class);
    }

    @Bean
    public AuditEntityDao auditEntityDao(ActiveObjects ao,
                                         TransactionTemplate transactionTemplate,
                                         AuditQueryMapper auditQueryMapper,
                                         AuditEntityMapper auditEntityMapper,
                                         AoAuditEntityMapper aoAuditEntityMapper,
                                         PropertiesProvider propertiesProvider
    ) {
        return new AoAuditEntityDao(ao, transactionTemplate, auditQueryMapper, auditEntityMapper, aoAuditEntityMapper,
                propertiesProvider);
    }

    @Bean
    public AuditQueryMapper auditQueryMapper() {
        return new AuditQueryMapper();
    }
}
