package com.atlassian.audit.file;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.sal.api.ApplicationProperties;
import io.atlassian.util.concurrent.LazyReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Clock;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Objects;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.time.Clock.systemDefaultZone;
import static java.time.ZonedDateTime.now;
import static java.util.Comparator.comparingInt;
import static java.util.Objects.requireNonNull;

/**
 * Manages the current Path(File) that should receive audit logs.
 * Allows you to configure the location and size limit.
 * With defaults set for rolling audit log.
 * <p>
 * Files will be rolled over when either the disk file size exceeds the limit, as such it is not a hard limit,
 * files may exceed the limit by the size of 1 line at most or
 * the datetime value used for filenames has changed since the last write
 */
public class RotatingFileManager implements Supplier<Path> {

    public static final String DEFAULT_FILE_EXTENSION = ".audit.log";

    private static final Logger log = LoggerFactory.getLogger(FileMessagePublisher.class);
    private static final DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyyMMdd");
    private final ApplicationProperties appProperties;
    private final String auditSubFolder;

    private final LazyReference<Path> fileDirectory = new LazyReference<Path>() {
        @Override
        protected Path create() throws Exception {
            try {
                return appProperties
                        .getLocalHomeDirectory()
                        .map(c -> c.resolve(auditSubFolder))
                        .orElseThrow(() -> new IllegalStateException("Unable to resolve local home directory"));
            } catch (Exception e) {
                log.error("Unable to determine audit log folder ", e);
                return Paths.get("").resolve(auditSubFolder);
            }
        }
    };
    private final Clock clock;
    private final CachingRetentionFileConfigService cachingRetentionFileConfigService;
    private Path currentFilePath;

    public RotatingFileManager(@Nonnull ApplicationProperties appProperties,
                               @Nonnull String auditSubFolder,
                               CachingRetentionFileConfigService cachingRetentionFileConfigService) {
        this(appProperties,
                auditSubFolder,
                systemDefaultZone(),
                cachingRetentionFileConfigService);
    }

    @VisibleForTesting
    public RotatingFileManager(ApplicationProperties appProperties,
                               @Nonnull String auditSubFolder,
                               @Nonnull Clock clock,
                               CachingRetentionFileConfigService cachingRetentionFileConfigService) {
        this.clock = requireNonNull(clock);
        this.appProperties = appProperties;
        this.auditSubFolder = auditSubFolder;
        this.cachingRetentionFileConfigService = cachingRetentionFileConfigService;
    }

    @Override
    @Nonnull
    public Path get() {
        if (currentFilePath == null || currentFilePath.toFile().length() > getFileSizeLimitB() || filePrefixChanged(currentFilePath)) {
            rotate();
        }
        return currentFilePath;
    }

    private boolean filePrefixChanged(Path currentFilePath) {
        String currentPrefix = getPrefixFromFileName(currentFilePath.getFileName().toString());
        String calculatedPrefix = now(clock).format(dateTimeFormatter);
        return !Objects.equals(currentPrefix, calculatedPrefix);
    }

    private void rotate() {
        if (!fileDirectory.get().toFile().exists() && !fileDirectory.get().toFile().mkdirs()) {
            log.error("Unable to make audit log folder '{}'", fileDirectory.get().toAbsolutePath());
        }

        currentFilePath = fileDirectory.get().resolve(buildFileName(fileDirectory.get()));

        //check overall file number
        try (Stream<Path> pathStream = Files.list(fileDirectory.get())) {
            List<File> fileList = pathStream
                    .filter(path -> path.toString().endsWith(DEFAULT_FILE_EXTENSION))
                    .map(Path::toFile)
                    .sorted() //Default comparator compares File name, format yyyyMMdd.xxxxx.{extension}
                    .collect(Collectors.toList());
            for (int i = 0; i < fileList.size() - getFileCountLimit() + 1; i++) {
                log.info("Total number of audit file exceeds {} , removing file {}", fileList.size(), fileList.get(i).getName());
                fileList.get(i).delete();
            }
        } catch (IOException e) {
            log.error("Fail to limit file count on {}", fileDirectory.get().toString(), e);
        }
    }

    private String buildFileName(Path fileHomePath) {
        // check for existing files for today, e.g. after a restart
        String todayFormatted = now(clock).format(dateTimeFormatter);
        File[] todaysAuditFiles = fileHomePath.toFile()
                .listFiles((dir, name) -> name != null && name.endsWith(DEFAULT_FILE_EXTENSION) && name.contains(todayFormatted));

        if (todaysAuditFiles == null || todaysAuditFiles.length == 0) {
            return String.format("%s.%05d%s", todayFormatted, 0, DEFAULT_FILE_EXTENSION);
        }

        File latestAuditFile = Stream.of(todaysAuditFiles)
                .max(comparingInt(file -> getIterationFromFileName(file.getName())))
                .orElse(null);

        if (latestAuditFile == null) {
            return String.format("%s.%05d%s", todayFormatted, 0, DEFAULT_FILE_EXTENSION);
        }

        return (latestAuditFile.length() < getFileSizeLimitB()) ? latestAuditFile.getName() : // append to last existing file
                String.format("%s.%05d%s", todayFormatted, getIterationFromFileName(latestAuditFile.getName()) + 1, DEFAULT_FILE_EXTENSION); //rotate to new file
    }

    private int getIterationFromFileName(final String fileName) {
        String[] parts = fileName.split("\\.");
        String iterationStr = parts[1];
        return Integer.parseInt(iterationStr);
    }

    private String getPrefixFromFileName(final String fileName) {
        String[] parts = fileName.split("\\.");
        return parts[0];
    }

    private long getFileSizeLimitB() {
        return cachingRetentionFileConfigService.getConfig().getMaxFileSizeB();
    }

    private int getFileCountLimit() {
        return cachingRetentionFileConfigService.getConfig().getMaxFileCount();
    }
}
