package com.atlassian.audit.file;

import com.atlassian.audit.analytics.RetentionFileConfigUpdatedEvent;
import com.atlassian.audit.coverage.SingleValueCache;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import static java.util.concurrent.TimeUnit.SECONDS;

/**
 * A wrapper around retention file configuration service which caches the current value.
 */
public class CachingRetentionFileConfigService implements InitializingBean, DisposableBean {

    private final SingleValueCache<AuditRetentionFileConfig> configCache;

    private final EventPublisher eventPublisher;

    public CachingRetentionFileConfigService(EventPublisher eventPublisher,
                                             AuditRetentionFileConfigService delegate,
                                             int expirationSeconds) {
        this.eventPublisher = eventPublisher;
        configCache = new SingleValueCache<>(delegate::getConfig, expirationSeconds, SECONDS);
    }

    /**
     * This gets called frequently at high volume by the RotatingFileManager to check the file limit.
     * Hence caching is implemented to mitigate this.
     * Caching for the database retention config is redundant as the volume is low (only retention scheduler needs it).
     */
    public AuditRetentionFileConfig getConfig() {
        return configCache.get();
    }

    /**
     * Please keep in mind that we invalidate the cache only when coverage was changed on the local node.
     * If somebody changes coverage on another DC node, the cache will know nothing about it and expire after timeout.
     * This is fine to have the lag - as long as cache expiration timeout is not large.
     * We don't use cluster events here because there's no X-product way to do them.
     */
    @EventListener
    public void onRetentionFileConfigUpdatedEvent(RetentionFileConfigUpdatedEvent event) {
        configCache.invalidate();
    }

    @Override
    public void afterPropertiesSet() {
        eventPublisher.register(this);
    }

    @Override
    public void destroy() {
        eventPublisher.unregister(this);
    }
}
