package com.atlassian.audit.csv;

import com.atlassian.audit.entity.AuditAttribute;
import com.atlassian.audit.entity.AuditEntity;
import com.atlassian.audit.entity.AuditResource;
import com.atlassian.audit.entity.ChangedValue;
import com.atlassian.sal.api.message.I18nResolver;
import com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.supercsv.encoder.DefaultCsvEncoder;
import org.supercsv.io.CsvMapWriter;
import org.supercsv.prefs.CsvPreference;

import javax.annotation.Nonnull;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.time.ZoneId;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;
import java.util.stream.Collectors;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

public class AuditCsvWriter implements Closeable {
    private static final String I18N_TIMESTAMP = "atlassian.audit.common.timestamp";
    private static final String I18N_AUTHOR = "atlassian.audit.common.author";
    private static final String I18N_AUTHOR_ID = "atlassian.audit.common.author.id";
    private static final String I18N_SUMMARY = "atlassian.audit.common.summary";
    private static final String I18N_SYSTEM = "atlassian.audit.common.system";
    private static final String I18N_CATEGORY = "atlassian.audit.common.category";
    private static final String I18N_OBJECTS_AFFECTED = "atlassian.audit.common.objects.affected";
    private static final String I18N_CHANGED_VALUE = "atlassian.audit.common.changed.value";
    private static final String I18N_SOURCE = "atlassian.audit.common.source";
    private static final String I18N_METHOD = "atlassian.audit.common.method";
    private static final String I18N_NODE = "atlassian.audit.common.node";
    private static final String I18N_EXTRA_ATTRIBUTES = "atlassian.audit.common.extra.attributes";

    private static final String I18N_OBJECTS_AFFECTED_CELL_NAME = "atlassian.audit.common.objects.affected.cell.name";
    private static final String I18N_OBJECTS_AFFECTED_CELL_TYPE = "atlassian.audit.common.objects.affected.cell.type";
    private static final String I18N_OBJECTS_AFFECTED_CELL_ID = "atlassian.audit.common.objects.affected.cell.id";
    private static final String I18N_OBJECTS_AFFECTED_CELL_URI = "atlassian.audit.common.objects.affected.cell.uri";

    private static final String I18N_CHANGED_VALUE_CELL_NAME = "atlassian.audit.common.changed.values.cell.name";
    private static final String I18N_CHANGED_VALUE_CELL_FROM = "atlassian.audit.common.changed.values.cell.from";
    private static final String I18N_CHANGED_VALUE_CELL_TO = "atlassian.audit.common.changed.values.cell.to";

    private static final String I18N_EXTRA_ATTRIBUTES_CELL_NAME = "atlassian.audit.common.extra.attributes.cell.name";
    private static final String I18N_EXTRA_ATTRIBUTES_CELL_VALUE = "atlassian.audit.common.extra.attributes.cell.value";

    private static final Logger log = LoggerFactory.getLogger(AuditCsvWriter.class);

    private final String[] headingsRef;

    private CsvMapWriter csvMapWriter;
    private I18nResolver resolver;

    public AuditCsvWriter(I18nResolver resolver, OutputStream stream) {
        headingsRef = new String[]{
                resolver.getText(I18N_TIMESTAMP),
                resolver.getText(I18N_AUTHOR),
                resolver.getText(I18N_AUTHOR_ID),
                resolver.getText(I18N_CATEGORY),
                resolver.getText(I18N_SUMMARY),
                resolver.getText(I18N_SYSTEM),
                resolver.getText(I18N_OBJECTS_AFFECTED),
                resolver.getText(I18N_CHANGED_VALUE),
                resolver.getText(I18N_SOURCE),
                resolver.getText(I18N_METHOD),
                resolver.getText(I18N_NODE),
                resolver.getText(I18N_EXTRA_ATTRIBUTES)
        };
        csvMapWriter = this.createCsvWriter(stream);
        this.resolver = resolver;
    }

    public void appendHeader() {

        try {
            csvMapWriter.writeHeader(headingsRef);
        } catch (IOException e) {
            log.error("Failed to write header to outputStream", e);
        }
    }

    /**
     * Parses {@link AuditEntity} and appends that data as a row using {@link CsvMapWriter}
     *
     * @param auditEntity Entity with data to be used
     */
    public void appendRow(@Nonnull AuditEntity auditEntity) {
        requireNonNull(auditEntity, "auditEntity");

        Map<String, String> csvLine = ImmutableMap.<String, String>builder()
                .put(headingsRef[0], auditEntity.getTimestamp().atZone(ZoneId.systemDefault()).toString())
                .put(headingsRef[1], ofNullable(auditEntity.getAuthor().getName()).orElse(""))
                .put(headingsRef[2], auditEntity.getAuthor().getId())
                .put(headingsRef[3], ofNullable(auditEntity.getAuditType().getCategory()).orElse(""))
                .put(headingsRef[4], ofNullable(auditEntity.getAuditType().getAction()).orElse(""))
                .put(headingsRef[5], ofNullable(auditEntity.getSystem()).orElse(""))
                .put(headingsRef[6], affectedObjectsToString(auditEntity.getAffectedObjects()))
                .put(headingsRef[7], changedValuesToString(auditEntity.getChangedValues()))
                .put(headingsRef[8], ofNullable(auditEntity.getSource()).orElse(""))
                .put(headingsRef[9], ofNullable(auditEntity.getMethod()).orElse(""))
                .put(headingsRef[10], ofNullable(auditEntity.getNode()).orElse(""))
                .put(headingsRef[11], extraAttributesToString(auditEntity.getExtraAttributes()))
                .build();

        try {
            csvMapWriter.write(csvLine, headingsRef);
        } catch (IOException e) {
            log.error("Failed to append row to outputStream", e);
        }
    }

    private CsvMapWriter createCsvWriter(OutputStream stream) {
        //We use a buffered writer for performance reasons
        //See https://docs.oracle.com/javase/7/docs/api/java/io/OutputStreamWriter.html
        return new CsvMapWriter(new BufferedWriter(new OutputStreamWriter(stream, StandardCharsets.UTF_8)),
                new CsvPreference.Builder(CsvPreference.EXCEL_PREFERENCE).useEncoder(new DefaultCsvEncoder())
                        .build());
    }

    private String affectedObjectsToString(List<AuditResource> affectedObjects) {
        return affectedObjects.stream().map(
                affectedObject -> {
                    StringJoiner joiner = new StringJoiner(" ");
                    joiner.add(resolver.getText(I18N_OBJECTS_AFFECTED_CELL_NAME, affectedObject.getName()));
                    joiner.add(resolver.getText(I18N_OBJECTS_AFFECTED_CELL_TYPE, affectedObject.getType()));
                    appendIfNotNull(joiner, I18N_OBJECTS_AFFECTED_CELL_ID, affectedObject.getId());
                    appendIfNotNull(joiner, I18N_OBJECTS_AFFECTED_CELL_URI, affectedObject.getUri());
                    return joiner.toString();
                }
        ).collect(Collectors.joining(", "));
    }

    private void appendIfNotNull(StringJoiner joiner, String key, String value) {
        if (value != null) {
            joiner.add(resolver.getText(key, value));
        }
    }

    private String changedValuesToString(List<ChangedValue> values) {
        return values.stream().map(
                changedValue -> {
                    StringJoiner joiner = new StringJoiner(" ");
                    joiner.add(resolver.getText(I18N_CHANGED_VALUE_CELL_NAME, changedValue.getKey()));
                    appendIfNotNull(joiner, I18N_CHANGED_VALUE_CELL_FROM, changedValue.getFrom());
                    appendIfNotNull(joiner, I18N_CHANGED_VALUE_CELL_TO, changedValue.getTo());

                    return joiner.toString();
                }
        ).collect(Collectors.joining(", "));
    }

    private String extraAttributesToString(Collection<AuditAttribute> values) {
        return values.stream()
                .sorted(Comparator.comparing(AuditAttribute::getName))
                .map(auditAttribute -> {
                    StringJoiner joiner = new StringJoiner(" ");
                    joiner.add(resolver.getText(I18N_EXTRA_ATTRIBUTES_CELL_NAME, auditAttribute.getName()));
                    appendIfNotNull(joiner, I18N_EXTRA_ATTRIBUTES_CELL_VALUE, auditAttribute.getValue());
                    return joiner.toString();
                })
                .collect(Collectors.joining(", "));
    }

    @Override
    public void close() throws IOException {
        csvMapWriter.close();
    }
}


