package com.atlassian.audit.csv;

import com.atlassian.audit.analytics.ExportEvent;
import com.atlassian.audit.api.AuditQuery;
import com.atlassian.audit.api.AuditSearchService;
import com.atlassian.audit.api.AuditService;
import com.atlassian.audit.entity.*;
import com.atlassian.audit.plugin.AuditPluginInfo;
import com.atlassian.audit.rest.DelegatedViewTypeProvider;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.sal.api.message.I18nResolver;
import com.atlassian.sal.api.web.context.HttpContext;

import static com.atlassian.audit.ao.service.AuditedSearchService.auditQueryToString;
import static com.atlassian.audit.entity.AuditType.fromI18nKeys;
import static com.atlassian.audit.entity.CoverageArea.AUDIT_LOG;
import static com.atlassian.audit.entity.CoverageLevel.BASE;

public class AuditCsvExportService {

    private static final AuditType AUDIT_LOG_EXPORTED = fromI18nKeys(AUDIT_LOG, BASE, "atlassian.audit.event.category.audit", "atlassian.audit.event.action.audit.exported").build();

    private final SelectiveExportLicenseChecker licenseChecker;
    private final AuditSearchService searchService;
    private final I18nResolver resolver;
    private final HttpContext httpContextSupplier;
    private final EventPublisher eventPublisher;
    private final AuditPluginInfo auditPluginInfo;
    private final AuditService auditService;
    private final DelegatedViewTypeProvider delegatedViewTypeProvider = new DelegatedViewTypeProvider();

    public AuditCsvExportService(SelectiveExportLicenseChecker licenseChecker, AuditSearchService searchService,
                                 I18nResolver resolver, HttpContext httpContextSupplier, EventPublisher eventPublisher,
                                 AuditPluginInfo auditPluginInfo, AuditService auditService) {
        this.licenseChecker = licenseChecker;
        this.searchService = searchService;
        this.resolver = resolver;
        this.httpContextSupplier = httpContextSupplier;
        this.eventPublisher = eventPublisher;
        this.auditPluginInfo = auditPluginInfo;
        this.auditService = auditService;
    }

    /**
     * Creates a AuditWriter which writes the results of the given query
     *
     * @param query query to filter results by
     * @return an instance of AuditWriter that can write the results of the given query
     */
    public AuditCsvExporter createExporter(AuditQuery query) throws LicenseException {
        if (isSelectiveExport(query) && !licenseChecker.allowSelectiveExport()) {
            throw new LicenseException("Attempted to selectively export without correct license to do so");
        }
        eventPublisher.publish(new ExportEvent(isSelectiveExport(query),
                delegatedViewTypeProvider.getDelegatedViewType(httpContextSupplier),
                auditPluginInfo.getPluginVersion()));
        auditService.audit(AuditEvent.builder(AUDIT_LOG_EXPORTED)
                .extraAttribute(AuditAttribute.fromI18nKeys("atlassian.audit.event.attribute.query", auditQueryToString(query, Integer.MAX_VALUE)).build())
                .build());
        return new AuditCsvExporter(searchService, query, resolver);
    }

    private boolean isSelectiveExport(AuditQuery query) {
        //We don't consider specifying resource type/id as a selective filter because this is always provided
        //in the delegated view
        return !query.getActions().isEmpty() ||
                !query.getCategories().isEmpty() ||
                query.getFrom().isPresent() ||
                query.getTo().isPresent() ||
                !query.getUserIds().isEmpty() ||
                query.getSearchText().isPresent();
    }
}
