package com.atlassian.audit.coverage;

import javax.annotation.concurrent.ThreadSafe;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

/**
 * Holds single value which is expired after write or can be manually invalidated.
 * Doesn't guarantee that cache will be refreshed only once per expiration period - only best effort.
 * Optimized for read scenario and avoids locking.
 */
@ThreadSafe
public class SingleValueCache<T> implements Supplier<T> {

    private final long expirationNanos;

    private final AtomicReference<ValueHolder<T>> currentReference = new AtomicReference<>();

    private final Supplier<Long> nanoTimeProvider;

    private final Function<T, T> accumulator;

    /**
     *  {@code delegate} will be invoked once cache expired, the returned value will replace the current value in cache.
     */
    public SingleValueCache(Supplier<T> delegate, long duration, TimeUnit unit) {
        this(oldValue -> delegate.get(), duration, unit, System::nanoTime);
    }

    /**
     *  {@code accumulator} will be invoked once cache expired, the returned value will replace the current value in cache.
     *  Existing value in cache will be passed to {@code accumulator} for aggregation or accumulation. Null value is passed for 1st time execution.
     *
     */
    public SingleValueCache(Function<T, T> accumulator, long duration, TimeUnit unit) {
        this(accumulator, duration, unit, System::nanoTime);
    }

    public SingleValueCache(Function<T, T> accumulator, long duration, TimeUnit unit, Supplier<Long> nanoTimeProvider) {
        this.accumulator = accumulator;
        this.expirationNanos = unit.toNanos(duration);
        this.nanoTimeProvider = nanoTimeProvider;
    }

    @Override
    public T get() {
        final long currentTimeNanos = nanoTimeProvider.get();
        final ValueHolder<T> currentValue = currentReference.get();
        if (currentValue == null || // if not initialized or invalidated
                currentTimeNanos >= currentValue.creationTimeNanos + expirationNanos) { // if expired
            final T newValue = requireNonNull(accumulator.apply(currentValue == null ? null : currentValue.value));
            // If somebody else succeeds in update, we don't care
            currentReference.compareAndSet(currentValue, new ValueHolder<>(newValue, currentTimeNanos));
            return newValue;
        }
        return currentValue.value;
    }

    public void invalidate() {
        currentReference.set(null);
    }

    private static class ValueHolder<T> {
        private final T value;
        private final long creationTimeNanos;

        public ValueHolder(T value, long creationTimeNanos) {
            this.value = value;
            this.creationTimeNanos = creationTimeNanos;
        }
    }
}
