package com.atlassian.audit.coverage;

import com.atlassian.audit.api.AuditCoverageConfigService;
import com.atlassian.audit.entity.AuditCoverageConfig;
import com.atlassian.audit.entity.CoverageArea;
import com.atlassian.audit.entity.EffectiveCoverageLevel;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.sal.api.pluginsettings.PluginSettings;
import com.atlassian.sal.api.pluginsettings.PluginSettingsFactory;
import com.google.common.collect.BiMap;
import com.google.common.collect.ImmutableBiMap;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.atlassian.audit.entity.EffectiveCoverageLevel.BASE;

/**
 * An implementation of {@link AuditCoverageConfigService} which uses {@link PluginSettings} to store audit configuration.
 */
public class SalAuditCoverageConfigService implements InternalAuditCoverageConfigService {

    private static final String COVERAGE_CONFIG_PREFIX = "com.atlassian.audit.plugin:audit-config:coverage:";

    private static final BiMap<EffectiveCoverageLevel, String> stringKeyByLevel = ImmutableBiMap.copyOf(
            Stream.of(EffectiveCoverageLevel.values())
                    .collect(Collectors.toMap(Function.identity(), EffectiveCoverageLevel::getKey)));

    private static final BiMap<CoverageArea, String> stringKeyByArea = ImmutableBiMap.copyOf(
            Stream.of(CoverageArea.values())
                    .collect(Collectors.toMap(Function.identity(), a -> a.toString().toLowerCase())));

    private final PluginSettingsFactory pluginSettingsFactory;
    private final EventPublisher eventPublisher;

    public SalAuditCoverageConfigService(PluginSettingsFactory pluginSettingsFactory, EventPublisher eventPublisher) {
        this.pluginSettingsFactory = pluginSettingsFactory;
        this.eventPublisher = eventPublisher;
    }

    @Override
    public AuditCoverageConfig getConfig() {
        final PluginSettings settings = pluginSettingsFactory.createGlobalSettings();
        return getCoverage(settings);
    }

    @Override
    public void updateConfig(AuditCoverageConfig config) {
        final PluginSettings settings = pluginSettingsFactory.createGlobalSettings();
        config.getLevelByArea().forEach((area, level) ->
                settings.put(getCoverageConfigurationKey(area), stringKeyByLevel.get(level)));
        eventPublisher.publish(new CoverageUpdatedEvent());
    }

    private AuditCoverageConfig getCoverage(PluginSettings settings) {
        final Map<CoverageArea, EffectiveCoverageLevel> levelByArea = new HashMap<>();
        for (CoverageArea area : CoverageArea.values()) {
            final String levelValue = (String) settings.get(getCoverageConfigurationKey(area));
            final EffectiveCoverageLevel level = Optional.ofNullable(levelValue)
                    .map(stringKeyByLevel.inverse()::get)
                    .orElse(BASE);

            levelByArea.put(area, level);
        }
        return new AuditCoverageConfig(levelByArea);
    }

    static String levelToString(EffectiveCoverageLevel level) {
        return stringKeyByLevel.get(level);
    }

    static String areaToString(CoverageArea area) {
        return stringKeyByArea.get(area);
    }

    private String getCoverageConfigurationKey(CoverageArea area) {
        return COVERAGE_CONFIG_PREFIX + stringKeyByArea.get(area);
    }
}
