package com.atlassian.audit.coverage;

import com.atlassian.audit.entity.AuditCoverageConfig;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventListenerRegistrar;
import com.atlassian.sal.api.license.LicenseChangedEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import static com.atlassian.audit.entity.EffectiveCoverageLevel.BASE;
import static com.google.common.collect.Maps.transformValues;
import static java.util.Objects.requireNonNull;

/**
 * This class listens to license change events and makes sure that Audit coverage will be properly changed if
 * the user downgrades to Server.
 */
public class LicenseDowngradeListener implements InitializingBean, DisposableBean {

    private static final Logger log = LoggerFactory.getLogger(LicenseDowngradeListener.class);

    private final ProductLicenseChecker licenseChecker;

    private final InternalAuditCoverageConfigService coverageConfigService;

    private final EventListenerRegistrar eventListenerRegistrar;

    public LicenseDowngradeListener(
            ProductLicenseChecker licenseChecker,
            InternalAuditCoverageConfigService coverageConfigService,
            EventListenerRegistrar eventListenerRegistrar) {
        this.licenseChecker = licenseChecker;
        this.coverageConfigService = coverageConfigService;
        this.eventListenerRegistrar = eventListenerRegistrar;
    }

    @Override
    public void afterPropertiesSet() {
        eventListenerRegistrar.register(this);
    }

    @EventListener
    public void onLicenseChange(LicenseChangedEvent licenseChangedEvent) {
        if (licenseChecker.isNotDcLicense()) {
            log.info("License downgrade to Server has been detected");
            final AuditCoverageConfig currentConfig = coverageConfigService.getConfig();
            final AuditCoverageConfig maybeDowngradedConfig = new AuditCoverageConfig(
                    transformValues(currentConfig.getLevelByArea(),
                            level -> requireNonNull(level).mostRestrictive(BASE)));
            coverageConfigService.updateConfig(maybeDowngradedConfig);
            log.info("Audit coverage has been updated after license downgrade to Server");
        }
    }

    @Override
    public void destroy() {
        eventListenerRegistrar.unregister(this);
    }
}
