package com.atlassian.audit.coverage;

import com.atlassian.audit.entity.AuditCoverageConfig;
import com.atlassian.audit.entity.CoverageLevel;
import com.atlassian.audit.entity.EffectiveCoverageLevel;

import static com.atlassian.audit.entity.EffectiveCoverageLevel.BASE;
import static com.google.common.collect.Maps.transformValues;
import static java.util.Objects.requireNonNull;

/**
 * Wrapper around config service which makes sure that Server license can never benefit from advanced coverage.
 */
public class LicenseAwareCoverageConfigService implements InternalAuditCoverageConfigService {

    private final ProductLicenseChecker licenseChecker;

    private final InternalAuditCoverageConfigService origin;

    public LicenseAwareCoverageConfigService(ProductLicenseChecker licenseChecker, InternalAuditCoverageConfigService origin) {
        this.licenseChecker = licenseChecker;
        this.origin = origin;
    }

    /**
     * Fail if somebody is trying to set restricted level. UI should prevent this from happening.
     */
    @Override
    public void updateConfig(AuditCoverageConfig config) {
        if (licenseChecker.isNotDcLicense()) {
            for (EffectiveCoverageLevel newLevel : config.getLevelByArea().values()) {
                if (newLevel.shouldAllow(CoverageLevel.ADVANCED)) {
                    throw new IllegalArgumentException("License doesn't allow to set this coverage level " + newLevel);
                }
            }
        }
        origin.updateConfig(config);
    }

    /**
     * If for some reason the service returns restricted levels, just reset them to BASE.
     */
    @Override
    public AuditCoverageConfig getConfig() {
        final AuditCoverageConfig config = origin.getConfig();
        if (licenseChecker.isNotDcLicense()) {
            return new AuditCoverageConfig(
                    transformValues(config.getLevelByArea(), level -> requireNonNull(level).mostRestrictive(BASE)));
        }
        return config;
    }

}
