package com.atlassian.audit.broker;

import com.atlassian.audit.entity.AuditAttribute;
import com.atlassian.audit.entity.AuditEntity;
import com.atlassian.audit.entity.AuditType;
import com.atlassian.audit.entity.ChangedValue;
import com.atlassian.audit.service.TranslationService;

import javax.annotation.Nonnull;
import java.util.Collection;
import java.util.List;

import static java.util.stream.Collectors.toList;

public class TranslatingAuditBroker implements InternalAuditBroker {

    private final TranslationService translationService;
    private final InternalAuditBroker delegatedBroker;

    public TranslatingAuditBroker(InternalAuditBroker delegatedBroker, TranslationService translationService) {
        this.delegatedBroker = delegatedBroker;
        this.translationService = translationService;
    }

    @Override
    public void audit(@Nonnull AuditEntity entity) {
        delegatedBroker.audit(translate(entity));
    }

    private AuditEntity translate(AuditEntity entity) {
        return new AuditEntity.Builder(entity).type(translate(entity.getAuditType()))
                                              .changedValues(translate(entity.getChangedValues()))
                                              .extraAttributes(translate(entity.getExtraAttributes()))
                                              .build();

    }

    private List<ChangedValue> translate(List<ChangedValue> changedValues) {
        return changedValues.stream()
                            .map(c -> ChangedValue.fromI18nKeys(c.getI18nKey())
                                                  .withKeyTranslation(translate(c.getI18nKey()))
                                                  .from(c.getFrom())
                                                  .to(c.getTo())
                                                  .build())
                            .collect(toList());
    }

    private Collection<AuditAttribute> translate(Collection<AuditAttribute> extraAttributes) {
        return extraAttributes.stream()
                              .map(a -> AuditAttribute.fromI18nKeys(a.getNameI18nKey(), a.getValue()).withNameTranslation(translate(a.getNameI18nKey())).build())
                              .collect(toList());
    }

    private AuditType translate(AuditType auditType) {
        return AuditType.builder(auditType).withCategoryTranslation(translate(auditType.getCategoryI18nKey()))
                                           .withActionTranslation(translate(auditType.getActionI18nKey())).build();
    }

    private String translate(String key) {
        return translationService.getSiteLocaleText(key);
    }
}
