package com.atlassian.audit.ao.service;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.audit.api.AuditEntityCursor;
import com.atlassian.audit.api.AuditQuery;
import com.atlassian.audit.api.AuditSearchService;
import com.atlassian.audit.api.AuditService;
import com.atlassian.audit.api.util.pagination.Page;
import com.atlassian.audit.api.util.pagination.PageRequest;
import com.atlassian.audit.entity.AuditAttribute;
import com.atlassian.audit.entity.AuditEntity;
import com.atlassian.audit.entity.AuditEvent;
import com.atlassian.audit.entity.AuditType;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Comparator;
import java.util.Optional;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;

import static com.atlassian.audit.entity.CoverageArea.AUDIT_LOG;
import static com.atlassian.audit.entity.CoverageLevel.BASE;
import static java.lang.String.format;

public class AuditedSearchService implements AuditSearchService {

    private static final AuditType AUDIT_EVENT_SEARCHED = AuditType.fromI18nKeys(AUDIT_LOG, BASE, "atlassian.audit.event.category.audit", "atlassian.audit.event.action.audit.search").build();

    private final AuditSearchService delegate;
    private final AuditService auditService;

    public AuditedSearchService(AuditSearchService delegate, AuditService auditService) {
        this.delegate = delegate;
        this.auditService = auditService;
    }

    @Nonnull
    @Override
    public Page<AuditEntity, AuditEntityCursor> findBy(@Nonnull AuditQuery query,
                                                       @Nonnull PageRequest<AuditEntityCursor> pageRequest,
                                                       int scanLimit) throws TimeoutException {
        Page<AuditEntity, AuditEntityCursor> page = delegate.findBy(query, pageRequest, scanLimit);
        auditService.audit(createSearchAuditEvent(query, page, scanLimit));
        return page;
    }

    @VisibleForTesting
    public AuditEvent createSearchAuditEvent(AuditQuery query, Page<AuditEntity, AuditEntityCursor> page, int scanLimit) {
        String queryStr = auditQueryToString(query, scanLimit);

        Optional<AuditEntity> minId = page.getValues().stream().min(Comparator.comparingLong(AuditEntity::getId));
        Optional<AuditEntity> maxId = page.getValues().stream().max(Comparator.comparingLong(AuditEntity::getId));
        Optional<AuditEntity> minTimestamp = page.getValues().stream().min(Comparator.comparing(AuditEntity::getTimestamp));
        Optional<AuditEntity> maxTimestamp = page.getValues().stream().max(Comparator.comparing(AuditEntity::getTimestamp));

        return AuditEvent.builder(AUDIT_EVENT_SEARCHED)
                .extraAttribute(AuditAttribute.fromI18nKeys("atlassian.audit.event.attribute.query", queryStr).build())
                .extraAttribute(AuditAttribute.fromI18nKeys("atlassian.audit.event.attribute.results", String.valueOf(page.getSize())).build())
                .extraAttribute(AuditAttribute.fromI18nKeys("atlassian.audit.event.attribute.timestamp", minTimestamp.isPresent() && maxTimestamp.isPresent() ?
                        format("%s - %s", minTimestamp.get().getTimestamp(), maxTimestamp.get().getTimestamp()) : "no results").build())
                .extraAttribute(AuditAttribute.fromI18nKeys("atlassian.audit.event.attribute.id", minId.isPresent() && maxId.isPresent() ?
                        format("%s - %s", minId.get().getId(), maxId.get().getId()) : "no results").build())
                .build();
    }

    public static String auditQueryToString(AuditQuery query, int scanLimit) {
        return new StringBuilder()
                .append(query.getFrom().isPresent() ? format("From : %s;", query.getFrom().get()) : "")
                .append(query.getTo().isPresent() ? format("To : %s;", query.getTo().get()) : "")
                .append(!query.getUserIds().isEmpty() ? format("UserIds : %s;", query.getUserIds()) : "")
                .append(!query.getResources().isEmpty() ? format("Resources : %s", query.getResources().toString()) : "")
                .append(!query.getCategories().isEmpty() ? format("Categories : %s;", query.getCategories()) : "")
                .append(!query.getActions().isEmpty() ? format("Actions : %s;", query.getActions()) : "")
                .append(scanLimit < Integer.MAX_VALUE ? format("ScanLimit : %s;", scanLimit) : "")
                .append(query.getSearchText().isPresent() ? format("Freetext : %s;", query.getSearchText().get()) : "").toString();
    }

    @Override
    public void stream(@Nonnull AuditQuery query, int offset, int limit, @Nonnull Consumer<AuditEntity> consumer) throws TimeoutException {
        delegate.stream(query, offset, limit, consumer);
    }

    @Override
    public long count(@Nullable AuditQuery query) throws TimeoutException {
        return delegate.count(query);
    }
}

