package com.atlassian.audit.ao.dao;

import com.atlassian.audit.entity.ChangedValue;
import org.codehaus.jackson.annotate.JsonCreator;
import org.codehaus.jackson.annotate.JsonProperty;
import org.codehaus.jackson.map.ObjectMapper;

import javax.annotation.Nonnull;
import java.io.IOException;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import static java.util.Objects.requireNonNull;

public class JacksonChangedValuesSerializer implements ChangedValuesSerializer {
    private final ObjectMapper objectMapper;

    public static class ChangedValueData {
        private final String i18nKey;
        private final String key;
        private final String from;
        private final String to;

        @JsonCreator
        public ChangedValueData(@JsonProperty("i18nKey") String i18nKey,
                                @JsonProperty("key") String key,
                                @JsonProperty("from") String from,
                                @JsonProperty("to") String to) {
            this.i18nKey = i18nKey == null ? key : i18nKey;
            this.key = key;
            this.from = from;
            this.to = to;
        }

        public String getKey() {
            return key;
        }

        public String getI18nKey() {
            return i18nKey;
        }

        public String getFrom() {
            return from;
        }

        public String getTo() {
            return to;
        }
    }

    public JacksonChangedValuesSerializer(ObjectMapper objectMapper) {
        this.objectMapper = requireNonNull(objectMapper);
    }

    @Override
    public List<ChangedValue> deserialize(@Nonnull String s) {
        try {
            return Stream.of(objectMapper.readValue(s, ChangedValueData[].class))
                    .map(x -> ChangedValue.fromI18nKeys(x.getI18nKey()).withKeyTranslation(x.getKey()).from(x.getFrom()).to(x.getTo()).build())
                    .collect(Collectors.toList());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public String serialize(@Nonnull Iterable<ChangedValue> changedValues) {
        try {
            ChangedValueData[] values = StreamSupport.stream(changedValues.spliterator(), false)
                    .map(x -> new ChangedValueData(x.getI18nKey(), x.getKey(), x.getFrom(), x.getTo()))
                    .toArray(ChangedValueData[]::new);
            return objectMapper.writeValueAsString(values);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
